package at.tugraz.genome.maspectras.quantification;

import at.tugraz.genome.util.BioUtilsConstants;
import at.tugraz.genome.util.index.IndexFileException;
import at.tugraz.genome.util.index.IndexedLineNumberHeaderReader;
import at.tugraz.genome.maspectras.GlobalConstants;

/**
 * Reads the header file for the chromatograms
 * 
 * @author Juergen Hartler
 */
public class ChromatogramHeaderFileReader extends IndexedLineNumberHeaderReader
{
  /**
   * constructor to read the header file
   * @param fileName the path to the header file
   */
  public ChromatogramHeaderFileReader(String fileName){
    super(fileName);
  }
  
  /**
   * 
   * @return the highest possible m/z value
   * @throws CgException
   */
  public int getHighestMz() throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    String mz = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_HIGHEST_MZ);
    if (mz==null) throw new CgException("There is no highest m/z Value");
    return new Integer(mz);
  }

  /**
   * 
   * @return the lowest possible m/z value
   * @throws CgException
   */
  public int getLowestMz() throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    String mz = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_LOWEST_MZ);
    if (mz==null) throw new CgException("There is no lowest m/z Value");
    return new Integer(mz);
  }

  /**
   * 
   * @return the multiplication factor to get integer lines from m/z values
   * @throws CgException
   */
  public int getMzMultiplicationFactor() throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    String mFactor = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MZ_MULTIPLICATION_FACTOR);
    if (mFactor==null) throw new CgException("There is no mzMultiplicationFactorForInt");
    return new Integer(mFactor);
  }
  
  /**
   * 
   * @return the resolution value
   * @throws CgException
   */
  public int getLowestResolution() throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    String lowestResolution = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_LOWEST_RESOLUTION);
    if (lowestResolution==null) throw new CgException("There is no lowest Resolution");
    return new Integer(lowestResolution);
  }  

  /**
   * 
   * @return the amount of scans (MS or MS/MS)
   * @throws CgException
   */
  public int getNumberOfScans()throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    String lowestResolution = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_NUMBER_SCANS);
    if (lowestResolution==null) throw new CgException("There is no number of scans");
    return new Integer(lowestResolution);
  }  

  public int getNumberOfScans(int msLevel)throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    try {
      checkMSLevel(msLevel);
    }catch (IndexFileException e){throw new CgException(e.getMessage());}
    if (msLevel==1) return this.getNumberOfScans();
    else {
      String numberOfScans = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_NUMBER_SCANS+String.valueOf(msLevel));
      if (numberOfScans==null) throw new CgException("There is no numberOfScans"+String.valueOf(msLevel)+" file path");
      return new Integer(numberOfScans);
    }
  }  

  
  /**
   * 
   * @return the path to the retention time file
   * @throws CgException
   */
  public String getRetentionTimeFile()throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    String retentionFile = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_RETENTION_TIME);
    if (retentionFile==null) throw new CgException("There is no retention file");
    return retentionFile;
  }
  
  public String getIndexFilePath (int msLevel) throws IndexFileException{
    if (properties_==null) throw new IndexFileException("The file must be read first");
    checkMSLevel(msLevel);
    if (msLevel==1) return super.getIndexFilePath();
    else {
      String indexFilePath = (String)properties_.get(BioUtilsConstants.INDEX_HEADER_FILE_INDEX_FILE+String.valueOf(msLevel));
      if (indexFilePath==null) throw new IndexFileException("There is no index"+String.valueOf(msLevel)+" file path");
      return indexFilePath;
    }
  }
  
  public String getIndexedFilePath(int msLevel) throws IndexFileException{
    if (properties_==null) throw new IndexFileException("The file must be read first");
    checkMSLevel(msLevel);
    if (msLevel==1) return super.getIndexedFilePath();
    else {
      String indexedFilePath = (String)properties_.get(BioUtilsConstants.INDEX_HEADER_FILE_INDEXED_FILE+String.valueOf(msLevel));
      if (indexedFilePath==null) throw new IndexFileException("There is no index"+String.valueOf(msLevel)+" file path");
      return indexedFilePath;
    }
  }
  
  public String getRetentionTimeFile(int msLevel)throws CgException{
    try {
      checkMSLevel(msLevel);
    }catch (IndexFileException e){throw new CgException(e.getMessage());}
    if (msLevel==1) return this.getRetentionTimeFile();
    else {
      String retentionFile = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_RETENTION_TIME+String.valueOf(msLevel));
      if (retentionFile==null) throw new CgException("There is no index"+String.valueOf(msLevel)+" file path");
      return retentionFile;
    }
  }
  
  public int getMsLevel()throws IndexFileException{
    if (properties_==null) throw new IndexFileException("The file must be read first");
    if (!properties_.containsKey(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MS_LEVEL))
      throw new IndexFileException("This chrom file is not capable of MS/MS");
    String msLevel = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MS_LEVEL);
    if (msLevel==null) throw new IndexFileException("There is no MS-Level");
    return Integer.parseInt(msLevel);
    
  }
  
  private void checkMSLevel(int msLevel) throws IndexFileException{
    if (properties_==null) throw new IndexFileException("The file must be read first");
    if (msLevel==1) return;
    else if (msLevel>1){
      if (!properties_.containsKey(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MS_LEVEL))
        throw new IndexFileException("The file does not support an MS-Level of \""+msLevel+"\"! The highest available one is 1!");
      int highestMsLevel = getMsLevel();
      if (msLevel>highestMsLevel)
        throw new IndexFileException("The file does not support an MS-Level of \""+msLevel+"\"! The highest available one is "+highestMsLevel+"!");
    }else throw new IndexFileException("The requested MS-Level of \""+msLevel+"\" is not supported!");
  }

  public String getMsMsType()throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    if (!properties_.containsKey(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MSMS_TYPE))
      throw new CgException("This chrom file is not capable of MS/MS");
    String msmsType = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MSMS_TYPE);
    if (msmsType==null) throw new CgException("There is no MS/MS-Type");
    return msmsType;
  }
  
  /**
   * 
   * @return the highest possible m/z value
   * @throws CgException
   */
  public int getHighestMz(int msLevel) throws CgException{
    try {
      checkMSLevel(msLevel);
    }catch (IndexFileException e){throw new CgException(e.getMessage());}
    if (msLevel==1) return this.getHighestMz();
    else {
      String mz = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_HIGHEST_MZ+String.valueOf(msLevel));
      if (mz==null) throw new CgException("There is no highest m/z value for MS-Level "+String.valueOf(msLevel)+"");
      return Integer.parseInt(mz);
    }
  }

  /**
   * 
   * @return the lowest possible m/z value
   * @throws CgException
   */
  public int getLowestMz(int msLevel) throws CgException{
    try {
      checkMSLevel(msLevel);
    }catch (IndexFileException e){throw new CgException(e.getMessage());}
    if (msLevel==1) return this.getHighestMz();
    else {
      String mz = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_LOWEST_MZ+String.valueOf(msLevel));
      if (mz==null) throw new CgException("There is no lowest m/z value for MS-Level "+String.valueOf(msLevel)+"");
      return Integer.parseInt(mz);
    }
  }
  
  /**
   * 
   * @return null when there is no polarity switched data, "positive" or "negative" otherwise
   * @throws CgException
   */
  public String getPolaritySwitched() throws CgException{
    if (properties_==null) throw new CgException("The file must be read first");
    String polarity = null;
    if (properties_.containsKey(GlobalConstants.CHROMATOGRAM_HEADER_FILE_POLARITY_SWITCHED)){
      polarity = (String)properties_.get(GlobalConstants.CHROMATOGRAM_HEADER_FILE_POLARITY_SWITCHED);
      if (!polarity.equalsIgnoreCase(GlobalConstants.CHROMATOGRAM_HEADER_FILE_POLARITY_POSITIVE) && !polarity.equalsIgnoreCase(GlobalConstants.CHROMATOGRAM_HEADER_FILE_POLARITY_NEGATIVE))
        throw new CgException("The polarity \""+polarity+"\" is not supported for polarity switched data");
    }
    return polarity;
  }

}
