/*
 * Created on 23.06.2005
 * Erik Koerner, FH Joanneum
 * erik.koerner@fh-joanneum.at
 */
package at.tugraz.genome.maspectras.quantification;

import java.io.*;
import java.nio.*;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import java.util.Vector;
import java.util.zip.GZIPInputStream;
import java.util.zip.Inflater;

import at.tugraz.genome.maspectras.utils.StringUtils;

/**
 * XPP3-based mzXML Reader for the conGenetics Package.
 * 
 * This class reads an mzXML file with hierarchical scans. It uses two methods
 * defined in the CgIAddScan interface to return created header and spectrum
 * information to the caller. Only level 1 spectra are read, spectra of higher
 * order are skipped, however, the schan number of thos spectra is remembered in
 * the scan objects.
 * 
 * @author Erik Koerner, FH Joanneum/ITM
 * @version 25.0.2005
 */
public class CgMzXmlReader implements CgReader
{
  XmlPullParser rdr; // my Parser...

  CgIAddScan adder;

  public boolean oneBaseScan = false;

  private boolean parseMsMs_;
  
  private boolean returnScansAnyWay_;

  private int lowestMz = 1000000 * CgDefines.mzMultiplicationFactorForInt;

  private int highestMz = 0;
  
  private float lowerThreshold = 0;
  
  private float upperThreshold = 1000000;
  
//  private FileReader fileReader;
  private InputStream inStream_;
  
  private int multiplicationFactorForInt_;
    
  CgScanHeader myHeader;

  /**
   * @param callbacks
   *          Pass an Object implementing this interface. The methods will be
   *          called whenever a msScan header or individual scans are generated
   *          during the reading process.
   */
  public CgMzXmlReader(CgIAddScan callbacks, boolean parseMsMs)
  {
    multiplicationFactorForInt_ = CgDefines.mzMultiplicationFactorForInt;
    adder = callbacks;
    oneBaseScan = true;
    this.parseMsMs_ = parseMsMs;
    lowestMz = 1000000 * CgDefines.mzMultiplicationFactorForInt;
  }
  
  public CgMzXmlReader(CgIAddScan callbacks, boolean parseMsMs, int multiplicationFactorForInt)
  {
    this(callbacks, parseMsMs);
    multiplicationFactorForInt_ = multiplicationFactorForInt;
    lowestMz = 1000000 * multiplicationFactorForInt_;  
  }
  
  public CgMzXmlReader(CgIAddScan callbacks, boolean parseMsMs, boolean returnScansAnyWay)
  {
    this (callbacks, parseMsMs);
    this.returnScansAnyWay_ = returnScansAnyWay;
    if (this.returnScansAnyWay_){
      this.parseMsMs_ = true;
    }
  }

  /**
   * Method to read in an mzXML file synchronously. This method returns after
   * reading in the complete mzXML file.
   * 
   * @param fileName
   *          File Name (including full path if required) for the mzXML file to
   *          be read
   * @throws CgException
   *           All internal exceptions are mapped to the CgException type.
   */
  public void ReadFile(String fileName) throws CgException
  {
    this.ReadFile(fileName,false); 
  }
  
  
  public void ReadFile(String fileName, boolean readJustMzMaxima) throws CgException
  {
    int eventType;
//    System.out.println("Lower/Upper: "+this.lowerThreshold+" ; "+this.upperThreshold);
    try {
      // =========================================================
      // Open the file:
      // =========================================================

      XmlPullParserFactory factory = XmlPullParserFactory.newInstance(System
          .getProperty(XmlPullParserFactory.PROPERTY_NAME), null);
      factory.setNamespaceAware(true);
      rdr = factory.newPullParser();
      File mzXMLFile = new File(fileName);
      if (mzXMLFile.exists()){
        inStream_ = new FileInputStream(mzXMLFile); 
      }else{
        mzXMLFile = new File(fileName+".gz");
        if (mzXMLFile.exists()){
          inStream_ = new GZIPInputStream(new FileInputStream(mzXMLFile)); 
        }else{
          throw new CgException("The file "+fileName+" does not exist!");
        }
      }
      rdr.setInput(inStream_,null);

      // =========================================================
      // Read the XML Data:
      // =========================================================

      eventType = rdr.getEventType();
      do {
        switch (eventType) {
          case XmlPullParser.START_DOCUMENT:
            break;
          case XmlPullParser.END_DOCUMENT:
            break;
          case XmlPullParser.START_TAG:
            if (rdr.getName() == "msRun")
              XmlReadMsRun(readJustMzMaxima);
            break;
        }
        eventType = rdr.next();
      } while (eventType != XmlPullParser.END_DOCUMENT);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      throw new CgException(ex.getMessage());
    }finally{
      try{
        inStream_.close();
        rdr.setInput(new StringReader("null"));
        rdr = null;
      }catch(IOException iox){
        iox.printStackTrace();
      }
      catch (XmlPullParserException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
      }
    }
  }
  
  

  /**
   * This Function reads a full MsRun Structure into Memory. Basi- cally it
   * reads each scan with msLevel=1, inner scans are re- gistered, but not fully
   * read.
   * 
   * @throws CgException
   */
  private void XmlReadMsRun(boolean readJustMzMaxima) throws CgException
  {
    int i;
    int eventType;
    

    myHeader = new CgScanHeader();
    myHeader.highestMSLevel = 1;
    // =========================================================
    // Read the msRun-Attributes. We put these into our Header
    // object.
    // =========================================================

    for (i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i) == "scanCount") {
        myHeader.ScanCount = Integer.parseInt(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "startTime") {
        myHeader.StartTime = XmlTimeIntervalToTime(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "endTime") {
        myHeader.EndTime = XmlTimeIntervalToTime(rdr.getAttributeValue(i));
      }
    }
    if (adder != null)
      adder.AddHeader(myHeader);
    else{
      if (!readJustMzMaxima)
        throw new CgException("No adder for Header and Scans defined.");
    }  

    // =========================================================
    // Finally read the bottom level scans.
    // =========================================================
    try {
      do {
        eventType = rdr.getEventType();
        switch (eventType) {
          case XmlPullParser.START_TAG:
            if (rdr.getName().equalsIgnoreCase("parentFile")){
              for (i = 0; i < rdr.getAttributeCount(); i++) {
                if (rdr.getAttributeName(i).equalsIgnoreCase("fileName")) {
                  adder.addParentFileName(StringUtils.getFileNameWOSuffix(rdr.getAttributeValue(i)));
                }
              }
            }else if (rdr.getName() == "scan"){
              if (readJustMzMaxima)
                xmlReadMaxima();
              else  
                XmlReadScan(null);
            }  
            break;
          case XmlPullParser.END_TAG:
            if (rdr.getName() == "msRun")
              return;
            break;
        }
        eventType = rdr.next();
      } while (eventType != XmlPullParser.END_DOCUMENT);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      throw new CgException(ex.getMessage());
    }
  }
  
  private void xmlReadMaxima() throws CgException
  {
    int i;
    int eventType;

    int msLevel = 0;
    float lowMz = 0;
    float highMz = 0;
    boolean lowMzFound = false;
    boolean highMzFound = false;
    int peaksCount = 0;

    for (i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i) == "msLevel") {
        msLevel = Integer.parseInt(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "lowMz") {
        lowMz = Float.parseFloat(rdr.getAttributeValue(i));
        lowMzFound = true;
      } else if (rdr.getAttributeName(i) == "highMz") {
        highMz = Float.parseFloat(rdr.getAttributeValue(i));
        highMzFound=true;
      } else if (rdr.getAttributeName(i) == "peaksCount") {
        peaksCount = Integer.parseInt(rdr.getAttributeValue(i));
  
      }

    }
    boolean foundMzBorders = false;
    if (lowMzFound&&highMzFound) foundMzBorders = true;

    try {
      eventType = rdr.next();
      do {
        switch (eventType) {
          case XmlPullParser.START_TAG:
            if (rdr.getName() == "peaks") {
              if (msLevel == 1) {
                oneBaseScan = true;
//                this.xmlReadMaxima();
                if (!foundMzBorders){
                  float[] maxima = readMaximaFromPeaks(peaksCount);
                  lowMz = maxima[0];
                  highMz = maxima[1];
                }
                int currentLowMz =  Math.round(lowMz*multiplicationFactorForInt_);
                int currentHighMz = Math.round(highMz*multiplicationFactorForInt_);                
                if (currentLowMz<this.lowestMz) this.lowestMz = currentLowMz ;
                if (currentHighMz>this.highestMz) this.highestMz = currentHighMz;
                // =================================================
                // Now we can inform our caller that we have a valid
                // new scan!
                // =================================================

              }
              if (parseMsMs_ && msLevel > myHeader.highestMSLevel) myHeader.highestMSLevel=msLevel;
            } else if (rdr.getName() == "scan") {
              xmlReadMaxima();
            }
            break;
          case XmlPullParser.END_TAG:
            if (rdr.getName() == "scan")
              return;
            break;
        }
        eventType = rdr.next();
      } while (eventType != XmlPullParser.END_DOCUMENT);
    }
    catch (Exception ex) {
      throw new CgException(ex.getMessage());
    }

  }

    
  /**
   * This Method reads a scan. It calls itself recursively in case scans are
   * element of our scan. However, if the scan level is > 1, the scan's content
   * is skipped for performance reasons.
   * 
   * @param scBase
   *          Pass the CgScan object that represents the level 1 scan to which
   *          this scan belongs to.
   * @throws CgException
   */
  private void XmlReadScan(CgScan scBase1) throws CgException
  {
    int i;
    int eventType;
    CgScan sc = null;
    CgScan scBase = scBase1;
    int num = 0;
    int msLevel = 0;
    int peaksCount = 0;
    float retentionTime = 0;
    float lowMz = 0;
    float highMz = 0;
    float basePeakMz = 0;
    float basePeakIntensity = 0;
    float totIonCurrent = 0;
    float precursorIntensity = 0;
    String precursorMz = "0";
    boolean lowMzFound = false;
    boolean highMzFound = false;
    String polarityString = "";
    int polarity = CgDefines.POLARITY_NO;


    // =========================================================
    // First of all we read the attributes:
    // =========================================================

    for (i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i) == "num") {
        num = Integer.parseInt(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "msLevel") {
        msLevel = Integer.parseInt(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "peaksCount") {
        peaksCount = Integer.parseInt(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "retentionTime") {
        retentionTime = XmlTimeIntervalToTime(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "lowMz") {
        lowMz = Float.parseFloat(rdr.getAttributeValue(i));
        lowMzFound = true;
      } else if (rdr.getAttributeName(i) == "highMz") {
        highMz = Float.parseFloat(rdr.getAttributeValue(i));
        highMzFound = true;
      } else if (rdr.getAttributeName(i) == "basePeakMz") {
        basePeakMz = Float.parseFloat(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "basePeakIntensity") {
        basePeakIntensity = Float.parseFloat(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "totIonCurrent") {
        totIonCurrent = Float.parseFloat(rdr.getAttributeValue(i));
      } else if (rdr.getAttributeName(i) == "polarity") {
        polarityString = rdr.getAttributeValue(i);
        if (polarityString.equalsIgnoreCase("+"))
          polarity = CgDefines.POLARITY_POSITIVE;
        else if (polarityString.equalsIgnoreCase("-"))
          polarity = CgDefines.POLARITY_NEGATIVE;
        else
          throw new CgException("The scan contains an unknown polarity \""+polarityString+"\" at scan number: "+num);
      }
    }
    boolean foundMzBorders = false;
    if (lowMzFound&&highMzFound) foundMzBorders = true;
    
    // =========================================================
    // Now we read the peaks:
    // =========================================================

    try {
      eventType = rdr.next();
      do {
        switch (eventType) {
          case XmlPullParser.START_TAG:
            if (rdr.getName() == "peaks") {
              if (msLevel == 1) {
                if (msLevel > myHeader.highestMSLevel) myHeader.highestMSLevel=msLevel;
                oneBaseScan = true;
                sc = new CgScan(peaksCount);
//                sc = new CgScan(0);
                sc.Num = num;
                sc.MsLevel = msLevel;
                sc.RetentionTime = retentionTime;
                sc.LowMz = lowMz;
                sc.HighMz = highMz;
                sc.BasePeakMz = basePeakMz;
                sc.BasePeakIntensity = basePeakIntensity;
                sc.TotIonCurrent = totIonCurrent;
                sc.setPolarity(polarity);
                XmlReadPeaks(sc,false,foundMzBorders);
                int currentLowMz = Math.round(sc.LowMz*multiplicationFactorForInt_);
                int currentHighMz = Math.round(sc.HighMz*multiplicationFactorForInt_);
                if (currentLowMz<this.lowestMz) this.lowestMz = currentLowMz ;
                if (currentHighMz>this.highestMz) this.highestMz = currentHighMz;
                // =================================================
                // Now we can inform our caller that we have a valid
                // new scan!
                // =================================================

                if (adder != null){
                  adder.AddScan(sc);
                  if (msLevel > myHeader.highestMSLevel) myHeader.highestMSLevel=msLevel;
                } else
                  throw new CgException(
                      "No adder for Header and Scans defined.");
              } else {
                if (scBase==null) scBase = adder.getLastBaseScan();
                if (scBase != null || this.returnScansAnyWay_) {
                  if (parseMsMs_) {
                    sc = new MsMsScan(peaksCount, num, msLevel, retentionTime,
                        lowMz, highMz, basePeakMz, basePeakIntensity,
                        totIonCurrent, precursorMz, precursorIntensity,polarity);
                    // System.out.println(num);
                    if (Float.valueOf(precursorMz)>this.lowerThreshold&&Float.valueOf(precursorMz)<this.upperThreshold){
                      XmlReadPeaks(sc,true,foundMzBorders);
                      if (scBase!=null)
                        scBase.AddSubscan(sc);
                      if (msLevel > myHeader.highestMSLevel) myHeader.highestMSLevel=msLevel;
                      if (this.returnScansAnyWay_){
                        this.adder.AddScan(sc);
                      }
                    }
                    // scBase.AddSubscanNumber(num);

                  } else
                    scBase.AddSubscanNumber(num);
                } else
                  throw new CgException("No base scan for subscan.");
              }
            } else if (parseMsMs_ && rdr.getName() == "precursorMz") {
              for (i = 0; i < rdr.getAttributeCount(); i++) {
                if (rdr.getAttributeName(i) == "precursorIntensity") {
                  precursorIntensity = Float.parseFloat(rdr
                      .getAttributeValue(i));
                  try {
                    rdr.next();
                    String childNode = rdr.getText().trim();
                    if (childNode != null) {
                      Float.parseFloat(childNode);
                      precursorMz = childNode;
                    }
                  }
                  catch (Exception ex) {
                    throw new CgException(ex.getMessage());
                  }
                }
              }
            } else if (rdr.getName() == "scan") {
              if (scBase != null)
                XmlReadScan(scBase);
              else
                XmlReadScan(sc);
            }
            break;
          case XmlPullParser.END_TAG:
            if (rdr.getName() == "scan")
              return;
            break;
        }
        eventType = rdr.next();
      } while (eventType != XmlPullParser.END_DOCUMENT);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      throw new CgException(ex.getMessage());
    }
  }

  /**
   * This Function reads the peaks of a single scan. If there is a valid CgScan
   * passed as parameter, it stores the values. If not, values are just skipped.
   * 
   * @param sc
   *          Pass the CgScan object that represents the level 1 scan to which
   *          this scan belongs to.
   * @throws CgException
   */
  private void XmlReadPeaks(CgScan sc, boolean msms, boolean foundMzBorders) throws CgException
  {
    int i, j;
    String s;
    CgBase64 cgb = new CgBase64();

    // =========================================================
    // Read the peaks - Attributes:
    // =========================================================
    String compressionType = null;
    
    if (sc != null) {
      for (i = 0; i < rdr.getAttributeCount(); i++) {
        if (rdr.getAttributeName(i) == "precision") {
          sc.Precision = Integer.parseInt(rdr.getAttributeValue(i));
        }
        if (rdr.getAttributeName(i) == "byteOrder") {
          sc.ByteOrder = rdr.getAttributeValue(i);
        } else if (rdr.getAttributeName(i) == "pairOrder") {
          sc.PairOrder = rdr.getAttributeValue(i);
        }  else if (rdr.getAttributeName(i) == "compressionType") {
          compressionType = rdr.getAttributeValue(i);
        }
      }
    }
    float lowestMzValue = Float.MAX_VALUE;
    float highestMzValue = 0f;
    try {
      rdr.next();
      s = rdr.getText().trim(); // In s we have a Base64 coded value array!
      if (sc == null || s == null || s.equalsIgnoreCase("</peaks>"))
        return;

      // =================================================
      // Process the data, in case we have to store it. In
      // C#, we have to store the byte array into a memory
      // stream and later on read it out float by float by
      // a binary reader.
      // =================================================
      byte[] decoded = cgb.decode(s);
      if (compressionType!=null && compressionType.equalsIgnoreCase("zlib")){
//        byte[] binArray = s.getBytes();
        Inflater decompressor = new Inflater();
        decompressor.setInput(decoded);
        ByteArrayOutputStream bos = null;
        try {
            bos = new ByteArrayOutputStream(decoded.length);

            // Decompress the data
            byte[] buf = new byte[1024];
            while (!decompressor.finished()) {
                int count = decompressor.inflate(buf);
                bos.write(buf, 0, count);
            }

        } finally {
            try {
                bos.close();
            } catch (Exception nope) { /* This exception doesn't matter */ }
        }
        decoded = bos.toByteArray();
        //s = new String(bos.toByteArray());
      }
      
      ByteBuffer byteBuf = ByteBuffer.wrap(decoded);
      Vector<Float> mzValues = new Vector<Float>();
      Vector<Float> intensities = new Vector<Float>();
      if (sc.Precision==64){
        double doubleArray[] = new double[decoded.length/8];
        DoubleBuffer doubleBuf = byteBuf.asDoubleBuffer();
        doubleBuf.get(doubleArray);
        j = 0;
        for (i = 0; i < sc.PeaksCount; i++) {
          float mzValue = (float)doubleArray[j++];
          float intensity = (float)doubleArray[j++];
          if (mzValue<lowestMzValue) lowestMzValue = mzValue;
          if (mzValue>highestMzValue) highestMzValue = mzValue;
          if ((mzValue>this.lowerThreshold&&mzValue<this.upperThreshold)||msms){
            mzValues.add(mzValue);
            intensities.add(intensity);
          }
        //sc.Scan[i][0] = floatArray[j++];
        //sc.Scan[i][1] = floatArray[j++];
        }        
      }else{
        float floatArray[] = new float[decoded.length / 4];
        FloatBuffer floatBuf = byteBuf.asFloatBuffer();
        floatBuf.get(floatArray);
        j = 0;
        for (i = 0; i < sc.PeaksCount; i++) {
          float mzValue = floatArray[j++];
          float intensity = floatArray[j++];
          if (mzValue<lowestMzValue) lowestMzValue = mzValue;
          if (mzValue>highestMzValue) highestMzValue = mzValue;
          if ((mzValue>this.lowerThreshold&&mzValue<this.upperThreshold)||msms){
            mzValues.add(mzValue);
            intensities.add(intensity);
          }
        //sc.Scan[i][0] = floatArray[j++];
        //sc.Scan[i][1] = floatArray[j++];
        }
      }
      sc.PeaksCount = mzValues.size();
      sc.Scan = new float[mzValues.size()][2];
      for (i=0; i!=mzValues.size();i++){
        sc.Scan[i][0] = mzValues.get(i);
        sc.Scan[i][1] = intensities.get(i);
      }
      if (sc.PeaksCount>0 && !foundMzBorders){
        sc.LowMz = lowestMzValue;
        sc.HighMz = highestMzValue;
      }
    }
    catch (Exception ex) {
      ex.printStackTrace();
      throw new CgException(ex.getMessage());
    }
  }
  
  private float[] readMaximaFromPeaks(int peaksCount) throws CgException{
    float[] maxima = new float[]{Float.MAX_VALUE,0f};
    if (peaksCount<1) return maxima;
    int precision = -1;
    String compressionType = null;
    CgBase64 cgb = new CgBase64();
    for (int i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i) == "precision") {
        precision = Integer.parseInt(rdr.getAttributeValue(i));
      }  else if (rdr.getAttributeName(i) == "compressionType") {
        compressionType = rdr.getAttributeValue(i);
      }
    }
    try {
      rdr.next();
      String s = rdr.getText().trim(); // In s we have a Base64 coded value array!
      if (s == null || s.equalsIgnoreCase("</peaks>"))
        return maxima;

      // =================================================
      // Process the data, in case we have to store it. In
      // C#, we have to store the byte array into a memory
      // stream and later on read it out float by float by
      // a binary reader.
      // =================================================
      byte[] decoded = cgb.decode(s);
      if (compressionType!=null && compressionType.equalsIgnoreCase("zlib")){
        Inflater decompressor = new Inflater();
        decompressor.setInput(decoded);
        ByteArrayOutputStream bos = null;
        try {
            bos = new ByteArrayOutputStream(decoded.length);
            byte[] buf = new byte[1024];
            while (!decompressor.finished()) {
                int count = decompressor.inflate(buf);
                bos.write(buf, 0, count);
            }

        } finally {
            try {
                bos.close();
            } catch (Exception nope) { /* This exception doesn't matter */ }
        }
        decoded = bos.toByteArray();
      }
      
      ByteBuffer byteBuf = ByteBuffer.wrap(decoded);
      float lowestMzValue = Float.MAX_VALUE;
      float highestMzValue = 0f;
      if (precision==64){
        double doubleArray[] = new double[decoded.length/8];
        DoubleBuffer doubleBuf = byteBuf.asDoubleBuffer();
        doubleBuf.get(doubleArray);
        lowestMzValue = (float)doubleArray[0];
        highestMzValue = (float)doubleArray[(peaksCount-1)*2];
      }else{
        float floatArray[] = new float[decoded.length / 4];
        FloatBuffer floatBuf = byteBuf.asFloatBuffer();
        floatBuf.get(floatArray);
        lowestMzValue = floatArray[0];
        highestMzValue = floatArray[(peaksCount-1)*2];
      }
      maxima[0] = lowestMzValue;
      maxima[1] = highestMzValue;
    }
    catch (Exception ex) {
      ex.printStackTrace();
      throw new CgException(ex.getMessage());
    }
    return maxima;
  }

  /**
   * Function converts an Time Interval to a double value[s]. Do something
   * better, this function is really very basic.
   * 
   * @param s
   *          String to parse
   * @return float representing a number of seconds
   */
  public static float XmlTimeIntervalToTime(String s)
  {
    if (s.startsWith("PT"))
      s = s.substring(2);
    if (s.endsWith("S"))
      s = s.substring(0, s.length() - 1);
    s = s.replace('e', 'E');
    return Float.parseFloat(s);
  }

  public int getHighestMz()
  {
    return this.highestMz;
  }

  public int getLowestMz()
  {
    return this.lowestMz;
  }

  public void setLowerThreshold(float lowerThreshold)
  {
    this.lowerThreshold = lowerThreshold;
  }

  public void setUpperThreshold(float upperThreshold)
  {
    this.upperThreshold = upperThreshold;
  }

  @Override
  public boolean usesPolaritySwitching()
  {
    // TODO Auto-generated method stub
    return false;
  }
  
  
}
