package at.tugraz.genome.maspectras.quantification;

import java.io.FileReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.Vector;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserFactory;
import org.xmlpull.v1.XmlPullParserException;

import at.tugraz.genome.dbutilities.SimpleValueObject;

import at.tugraz.genome.maspectras.quantification.vos.LevelRangeVO;
import at.tugraz.genome.maspectras.quantification.vos.CgMzDataDescriptionVO;
import at.tugraz.genome.maspectras.quantification.vos.CgMzPeakOrIntensityValueVO;

/**
 * Parser for mzData files
 * 
 * @author Juergen Hartler
 * 
 */
public class CgMzDataReader implements CgReader
{
  /** the xml reader */
  private XmlPullParser rdr; // my Parser...

  /** interface to add scans */
  private CgIAddScan adder;

  /** start time of the scans */
  private float startTime;

  /** stop time of the scans */
  private float stopTime;

  /** number of scans */
  private int scanCount;

  /** last parsed scan */
  private CgScan lastScan;
  /** the lowest mz value in the file*/
  private int lowestMz = 1000000 * CgDefines.mzMultiplicationFactorForInt;
  /** the highest mz value in the file*/
  private int highestMz = 0;
  /** the lower mzThreshold*multiplicationFactorForInt to translate the raw file in sections*/
  private float lowerThreshold = 0;
  /** the upper mzThreshold*multiplicationFactorForInt to translate the raw file in sections*/
  private float upperThreshold = 1000000;

  private FileReader fileReader;
  
  private int multiplicationFactorForInt_;
  
  private CgScanHeader myHeader;
  
  /**
   * @param callbacks
   *          interface to add scans
   */
  public CgMzDataReader(CgIAddScan callbacks)
  {
    adder = callbacks;
    multiplicationFactorForInt_ = CgDefines.mzMultiplicationFactorForInt;
    lowestMz = 1000000 * CgDefines.mzMultiplicationFactorForInt;
  }
  
  public CgMzDataReader(CgIAddScan callbacks, int multiplicationFactorForInt)
  {
    this(callbacks);
    multiplicationFactorForInt_ = multiplicationFactorForInt;
    lowestMz = 1000000 * multiplicationFactorForInt_;
  }

  /**
   * Reads in the mzData file
   * 
   * @param fileName
   *          absolute path to the file
   */
  public void ReadFile(String fileName) throws CgException
  {
    this.ReadFile(fileName,false); 
  }
  
  /**
   * Reads in the mzData file
   * 
   * @param fileName absolute path to the file
   * @param readJustMzMaxima should only the mzMaxima be read and the rest should be negelected?
   */
  public void ReadFile(String fileName, boolean readJustMzMaxima) throws CgException
  {
    int eventType;

    try {
      // =========================================================
      // Open the file:
      // =========================================================

      XmlPullParserFactory factory = XmlPullParserFactory.newInstance(System
          .getProperty(XmlPullParserFactory.PROPERTY_NAME), null);
      factory.setNamespaceAware(true);
      rdr = factory.newPullParser();
      rdr.setInput(new FileReader(fileName));
      fileReader = new FileReader(fileName);
      rdr.setInput(fileReader);
      // =========================================================
      // Read the XML Data:
      // =========================================================

      eventType = rdr.getEventType();
      do {
        switch (eventType) {
          case XmlPullParser.START_DOCUMENT:
            break;
          case XmlPullParser.END_DOCUMENT:
            break;
          case XmlPullParser.START_TAG:
            if (rdr.getName().equalsIgnoreCase("spectrumList"))
              xmlReadMsRun(readJustMzMaxima);
            break;
        }
        eventType = rdr.next();
      } while (eventType != XmlPullParser.END_DOCUMENT);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      throw new CgException(ex.getMessage());
    }finally{
      try{
        fileReader.close();
        rdr.setInput(new StringReader("null"));
        rdr = null;
      }catch(IOException iox){
        iox.printStackTrace();
      }
      catch (XmlPullParserException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
      }
    }
  }

  /**
   * Reads the contents of the tag spectrumList
   * @param readJustMzMaxima should only the mzMaxima be read and the rest should be negelected?
   * @throws CgException
   */
  private void xmlReadMsRun(boolean readJustMzMaxima) throws CgException
  {
    int i;
    int eventType;
    

    myHeader = new CgScanHeader();
    myHeader.highestMSLevel = 1;
    // =========================================================
    // Read the msRun-Attributes. We put these into our Header
    // object.
    // =========================================================

    for (i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i).equalsIgnoreCase("count")) {
        myHeader.ScanCount = Integer.parseInt(rdr.getAttributeValue(i));
        scanCount = myHeader.ScanCount;
      }/*
         * else if (rdr.getAttributeName(i)=="startTime") { myHeader.StartTime =
         * XmlTimeIntervalToTime(rdr.getAttributeValue(i)); } else if
         * (rdr.getAttributeName(i)=="endTime") { myHeader.EndTime =
         * XmlTimeIntervalToTime(rdr.getAttributeValue(i)); }
         */
    }

    if (adder != null)
      adder.AddHeader(myHeader);
    else
      if (!readJustMzMaxima)
        throw new CgException("No adder for Header and Scans defined.");

    // =========================================================
    // Finally read the bottom level scans.
    // =========================================================
    try {
      do {
        eventType = rdr.getEventType();
        switch (eventType) {
          case XmlPullParser.START_TAG:
            if (rdr.getName().equalsIgnoreCase("spectrum"))
              if (readJustMzMaxima)
                this.xmlReadMaxima();
              else  
                xmlReadScan(/* null */);
            break;
          case XmlPullParser.END_TAG:
            if (rdr.getName().equalsIgnoreCase("spectrumList")) {
              myHeader.StartTime = this.startTime;
              myHeader.EndTime = this.stopTime;

              System.out.println(myHeader.ScanCount);
              System.out.println(myHeader.StartTime);
              System.out.println(myHeader.EndTime);

              if (adder != null)
                adder.setStartStopHeader(myHeader);
              else
                if (!readJustMzMaxima)
                  throw new CgException("No adder for Header and Scans defined.");
              return;
            }
            break;
        }
        eventType = rdr.next();
      } while (eventType != XmlPullParser.END_DOCUMENT);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      throw new CgException(ex.getMessage());
    }

  }
  
  /**
   * reads only the maxima from the scans
   * @throws CgException
   * @throws XmlPullParserException
   * @throws IOException
   */
  private void xmlReadMaxima() throws CgException,XmlPullParserException, IOException
  {

    System.out.println("I am reading maxima");
    int i;
    int num = 0;
    /* int eventType; */
    int eventType;
    CgMzDataDescriptionVO descrVO = null;
    for (i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i).equalsIgnoreCase("id")) {
        num = Integer.parseInt(rdr.getAttributeValue(i));
      }
    }
    eventType = rdr.next();

    do {
      eventType = rdr.getEventType();
      rdr.getName();
      switch (eventType) {
        case XmlPullParser.START_TAG: {
          if (rdr.getName().equalsIgnoreCase("spectrumDesc"))
            descrVO = xmlReadDescription(num);
          break;
        }
        case XmlPullParser.END_TAG:
          if (rdr.getName().equalsIgnoreCase("spectrum")) {
            if (descrVO.getLevel() == 1) {
              if (descrVO.getLowerMz()<this.lowestMz) this.lowestMz = Math.round(descrVO.getLowerMz()*multiplicationFactorForInt_);
              if (descrVO.getUpperMz()>this.highestMz) this.highestMz = Math.round(descrVO.getUpperMz()*multiplicationFactorForInt_);

            }
            return;
          }
          break;
      }
      eventType = rdr.next();
    } while (eventType != XmlPullParser.END_DOCUMENT);
  }
  

  /**
   * Reads the contents of the spectrum tag; it does not matter of which MS level the scan originates
   * @throws CgException
   * @throws XmlPullParserException
   * @throws IOException
   */
  private void xmlReadScan() throws CgException,
      XmlPullParserException, IOException
  {
    int i;
    /* int eventType; */
    CgScan sc = null;
    int num = 0;
    int eventType;
    CgMzDataDescriptionVO descrVO = null;
    CgMzPeakOrIntensityValueVO mzValues = null;
    CgMzPeakOrIntensityValueVO intensityValues = null;
    /*
     * int peaksCount = 0;
     * 
     * 
     * 
     * 
     */
    // =========================================================
    // First of all we read the attributes:
    // =========================================================
    for (i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i).equalsIgnoreCase("id")) {
        num = Integer.parseInt(rdr.getAttributeValue(i));
        // System.out.println(num);
      }
    }
    eventType = rdr.next();

    do {
      eventType = rdr.getEventType();
      rdr.getName();
      switch (eventType) {
        case XmlPullParser.START_TAG: {
          if (rdr.getName().equalsIgnoreCase("spectrumDesc"))
            descrVO = xmlReadDescription(num);
          if (rdr.getName().equalsIgnoreCase("mzArrayBinary")) {
            if (descrVO != null) {
              /*
               * System.out.println("The descr VO!=null");
               * System.out.println(descrVO.getLevel());
               * System.out.println("ScanCount: "+num);
               */
              if (descrVO.getLevel() == 1) {
                mzValues = xmlReadByteData();
                /*
                 * if (sc!=null) sc = new CgScan(peaksCount); sc.Num = num;
                 * sc.MsLevel = msLevel; sc.RetentionTime = retentionTime;
                 * sc.LowMz = lowMz; sc.HighMz = highMz; sc.BasePeakMz =
                 * basePeakMz; sc.BasePeakIntensity = basePeakIntensity;
                 * sc.TotIonCurrent = totIonCurrent; XmlReadPeaks(sc);
                 */

                /*
                 * if (num==1){ System.out.println("MzArray:
                 * "+mzValues.getValues().length); for (int j=0;
                 * j!=mzValues.getValues().length;j++){ System.out.println(i+":
                 * "+mzValues.getValues()[j]); } }
                 */
              }
            }
          }
          if (rdr.getName().equalsIgnoreCase("intenArrayBinary")) {
            if (descrVO != null) {
              if (descrVO.getLevel() == 1) {
                intensityValues = xmlReadByteData();
              }

            }
          }
          /*
           * if (rdr.getName()=="scan") { if (scBase!=null) xmlReadScan(scBase);
           * else xmlReadScan(sc); }
           */

          break;
        }
        case XmlPullParser.END_TAG:
          if (rdr.getName().equalsIgnoreCase("spectrum")) {
            if (descrVO.getLevel() == 1) {
              Vector<Float> mzs = new Vector<Float>();
              Vector<Float> intensities = new Vector<Float>();
              for (int j = 0; j != mzValues.getLength(); j++) {
                float mzValue = mzValues.getValues()[j];
                if (mzValue>this.lowerThreshold&&mzValue<this.upperThreshold){
                  mzs.add(mzValue);
                // System.out.println(j);
                  intensities.add(intensityValues.getValues()[j]);
                }
              }             
              sc = new CgScan(mzs.size());
              sc.Num = num;
              sc.MsLevel = descrVO.getLevel();
              if (sc.MsLevel>myHeader.highestMSLevel) myHeader.highestMSLevel = sc.MsLevel;
              sc.RetentionTime = descrVO.getRetentionTime();
              sc.LowMz = descrVO.getLowerMz();
              sc.HighMz = descrVO.getUpperMz();
              sc.BasePeakMz = descrVO.getBasePeakMz();
              sc.BasePeakIntensity = descrVO.getBasePeakIntensity();
              sc.TotIonCurrent = descrVO.getTotIonCurrent();
              for (int j = 0; j != mzs.size(); j++) {
                sc.Scan[j][0] = mzs.get(j);
                // System.out.println(j);
                sc.Scan[j][1] = intensities.get(j);
              }
              if (descrVO.getLowerMz()<this.lowestMz) this.lowestMz = Math.round(descrVO.getLowerMz()*multiplicationFactorForInt_);
              if (descrVO.getUpperMz()>this.highestMz) this.highestMz = Math.round(descrVO.getUpperMz()*multiplicationFactorForInt_);
              adder.AddScan(sc);
              this.lastScan = sc;
            } else {
              if (lastScan != null)
                lastScan.AddSubscanNumber(num);
              else
                throw new CgException("No base scan for subscan.");
            }
            return;
          }
          break;
      }
      eventType = rdr.next();
    } while (eventType != XmlPullParser.END_DOCUMENT);
  }

  /**
   * Parses the contents of the tag spectrumDesc; here general information about the run is stored
   * @param scanNumber the actual scan number
   * @return the description VO
   * @throws XmlPullParserException
   * @throws IOException
   */
  private CgMzDataDescriptionVO xmlReadDescription(int scanNumber)
      throws XmlPullParserException, IOException
  {

    LevelRangeVO rangeVO = null;
    float retentionTime = 0;
    float basePeakMz = 0;
    float basePeakIntensity = 0;
    float totIonCurrent = 0;

    int eventType;
    eventType = rdr.getEventType();
    do {
      eventType = rdr.getEventType();
      switch (eventType) {
        case XmlPullParser.START_TAG: {
          if (rdr.getName().equalsIgnoreCase("spectrumInstrument")) {
            rangeVO = xmlReadLevelPlusRange();
            /*
             * if (scanNumber==1){ System.out.println(rangeVO.getLevel());
             * System.out.println(rangeVO.getLowerMz());
             * System.out.println(rangeVO.getUpperMz()); }
             */
          }
          if (rdr.getName().equalsIgnoreCase("cvParam")) {
            SimpleValueObject svo = xmlReadCvParam();
            if (svo.getLabel().equalsIgnoreCase("TimeInMinutes")) {
              retentionTime = CgMzXmlReader.XmlTimeIntervalToTime(svo
                  .getValue());
              if (scanNumber == 1) {
                startTime = retentionTime;
              }
              if (scanNumber == scanCount) {
                stopTime = retentionTime;
              }
              retentionTime = retentionTime * 60;
              if (scanNumber == scanCount) {
                System.out.println(retentionTime);
              }
            }

            /*
             * if (scanNumber==1){ System.out.println(rangeVO.getLevel());
             * System.out.println(rangeVO.getLowerMz());
             * System.out.println(rangeVO.getUpperMz()); }
             */
          }
          if (rdr.getName().equalsIgnoreCase("userParam")) {
            SimpleValueObject svo = xmlReadCvParam();
            if (svo.getLabel().equalsIgnoreCase("mzXML:basePeakMz")) {
              basePeakMz = Float.parseFloat(svo.getValue());
            }
            if (svo.getLabel().equalsIgnoreCase("mzXML:basePeakIntensity")) {
              basePeakIntensity = Float.parseFloat(svo.getValue());
            }
            if (svo.getLabel().equalsIgnoreCase("mzXML:totIonCurrent")) {
              totIonCurrent = Float.parseFloat(svo.getValue());
            }
          }

          break;
        }
        case XmlPullParser.END_TAG:
          if (rdr.getName().equalsIgnoreCase("spectrumDesc")) {
            CgMzDataDescriptionVO vo = new CgMzDataDescriptionVO(rangeVO,
                retentionTime, basePeakMz, basePeakIntensity, totIonCurrent);
            return vo;
          }
          break;
      }
      eventType = rdr.next();
    } while (eventType != XmlPullParser.END_DOCUMENT);
    return null;
  }

  /**
   * @return the MS level and the m/z range of one spectrum
   */
  private LevelRangeVO xmlReadLevelPlusRange()
  {

    int i = 0;

    int msLevel = 0;
    float lowMz = 0;
    float highMz = 0;

    for (i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i).equalsIgnoreCase("msLevel")) {
        msLevel = Integer.parseInt(rdr.getAttributeValue(i));
      }
      if (rdr.getAttributeName(i).equalsIgnoreCase("mzRangeStart")) {
        lowMz = Float.parseFloat(rdr.getAttributeValue(i));
      }
      if (rdr.getAttributeName(i).equalsIgnoreCase("mzRangeStop")) {
        highMz = Float.parseFloat(rdr.getAttributeValue(i));
      }
    }
    LevelRangeVO levelRangeVO = new LevelRangeVO(msLevel, lowMz, highMz);
    return levelRangeVO;
  }

  /**
   * @return the name value construct in the mzData file
   */
  private SimpleValueObject xmlReadCvParam()
  {

    int i = 0;

    String name = "";
    String value = "";

    for (i = 0; i < rdr.getAttributeCount(); i++) {
      if (rdr.getAttributeName(i).equalsIgnoreCase("name")) {
        name = rdr.getAttributeValue(i);
      }
      if (rdr.getAttributeName(i).equalsIgnoreCase("value")) {
        value = rdr.getAttributeValue(i);
      }
    }
    SimpleValueObject svo = new SimpleValueObject(name, value);
    return svo;
  }

  /**
   * @return reads the Base64 encoded string with the MS spectrum (m/z values or intensity array)
   * @throws XmlPullParserException
   * @throws IOException
   */
  private CgMzPeakOrIntensityValueVO xmlReadByteData()
      throws XmlPullParserException, IOException
  {

    int eventType = rdr.getEventType();
    CgMzPeakOrIntensityValueVO vo = null;
    do {
      eventType = rdr.getEventType();
      switch (eventType) {
        case XmlPullParser.START_TAG: {
          if (rdr.getName().equalsIgnoreCase("data")) {
            int length = 0;
            int precision = 0;
            String byteOrder = null;

            for (int i = 0; i < rdr.getAttributeCount(); i++) {
              if (rdr.getAttributeName(i) == "precision") {
                precision = Integer.parseInt(rdr.getAttributeValue(i));
              }
              if (rdr.getAttributeName(i) == "endian") {
                byteOrder = rdr.getAttributeValue(i);
              } else if (rdr.getAttributeName(i) == "length") {
                length = Integer.parseInt(rdr.getAttributeValue(i));
              }
            }

            if (length > 0 && precision > 0 && byteOrder != null
                && byteOrder.length() > 0) {
              vo = new CgMzPeakOrIntensityValueVO(precision, byteOrder, length);

              rdr.next();

              String s = rdr.getText().trim(); // In s we have a Base64 coded
                                                // value array!
              vo.setData(s);
              rdr.next();
              return vo;
            }

          }
          break;
        }
        case XmlPullParser.END_TAG:
          if (rdr.getName().equalsIgnoreCase("data")) {
            return null;
            // CgMzDataDescriptionVO vo= new
            // CgMzDataDescriptionVO(rangeVO,retentionTime,basePeakMz,basePeakIntensity,totIonCurrent);
            // return vo;
          }
          break;
      }
      eventType = rdr.next();
    } while (eventType != XmlPullParser.END_DOCUMENT);
    return null;
  }

  public int getHighestMz()
  {
    return this.highestMz;
  }

  public int getLowestMz()
  {
    return this.lowestMz;
  }

  public void setLowerThreshold(float lowerThreshold)
  {
    this.lowerThreshold = lowerThreshold;
  }

  public void setUpperThreshold(float upperThreshold)
  {
    this.upperThreshold = upperThreshold;
  }

  @Override
  public boolean usesPolaritySwitching()
  {
    // TODO Auto-generated method stub
    return false;
  }
}
