/*
 * Created on 24.06.2005
 * Erik Koerner, FH Joanneum
 * erik.koerner@fh-joanneum.at
 */
package at.tugraz.genome.maspectras.quantification;

import java.io.*;
import java.util.Hashtable;
import java.util.Vector;

import at.tugraz.genome.maspectras.utils.Calculator;

/**
 * Core number crunching class for Chromatogramm handling. 
 * Does all number crunching for peak analysis. Numerical functions mainly ported from
 * the ASAPRatio project (Savitzky Golay filtering, peak & valley detection, and background 
 * determination).
 * 
 * @author Erik Koerner
 * @version 24.06.2005
 */
public class CgChromatogram
{
    public float			LowerMzBand;			// Lower m/z area
    public float			Mz;						// m/z value
    public float			UpperMzBand;			// Upper m/z area
    
    public int				ScanCount;				// Number of scans
    public float			Value[][];				// The Chromatogram...
    
    
    public int				LoValley;				// Index of lower valley
    public int				Peak;					// Index of peak;	
    public int				UpValley;				// Index of upper valley
    
    public float			PeakTime;				// Retention Time of Peak
    public float			PeakTimeAverage;		// Retention Time of Peak averaged
    
    
    public float			Background;				// The background
    public float			Area;					// The area of our peak
    public float			AreaErr;				// The error of the area
    public boolean			Good;					// Does the area make sense
    public boolean anythingThere;// Are intensities at the position I am looking for a peak
    
    
    protected float			m_peakValue;			// Chroma peak (internal)
    protected float			m_average;				// Chroma average (internal)
    
    protected int startPosition_;
            
    public boolean isProfile_;
    
    protected float highestIntensity_ = 0f;
    
    public int startSmoothScan_ = -1;
    public int stopSmoothScan_ = -1;
    
    public float[] precalcPows_;

    

    public CgChromatogram(int size)
    {
        ScanCount = size;
        Value = new float[size][4];
        highestIntensity_ = 0f;
    }
    
    public void preSmooth(){
      float highestValue = 0f;
      for (int i=0;i!=ScanCount;i++){
        if (highestValue<Value[i][1])
          highestValue = Value[i][1];
        Value[i][2] = Value[i][1];
      }
      for (int i=0;i!=ScanCount;i++){
        if (Value[i][2]>0)
          Value[i][3] = doPreSmoothOperation(i,highestValue);
      }
      for (int i=0;i!=ScanCount;i++){
        Value[i][1] =  Value[i][3];
        Value[i][2] = Value[i][3];
      }
    }
    
    private float doPreSmoothOperation(int i, float highestValue){
      float smoothedValue = Value[i][2];
      if (Value[i][2]>0){
        float lowerValue = 0f;
        float upperValue = 0f;
        boolean lowerValueOverThresh = false;
        boolean upperValueOverThresh = false;
        for (int j=(i-1);(j!=(i-6))&&(j!=-1);j--){
          if (Value[j][2]>0){
            lowerValue = Value[j][2];
            if (Value[j][2]>highestValue*0.3){
              lowerValueOverThresh = true;
            }
            break;
          }
        }
        for (int j=(i+1);(j!=(i-6))&&(j!=ScanCount);j++){
          if (Value[j][2]>0){
            upperValue = Value[j][2];
            if (Value[j][2]>highestValue*0.3){
              upperValueOverThresh = true;
            }
            break;
          }
        }
        if (lowerValue>Value[i][2]&&upperValue>Value[i][2]&&
            (lowerValueOverThresh||upperValueOverThresh)){
          smoothedValue = (lowerValue+upperValue)/2;
        }
      }
      return smoothedValue;
    }
    
    public void smoothMean(float range, int repeats, boolean copyRawDataFirst){
      if (copyRawDataFirst) copyRawData();
      int startScan = 0;
      int stopScan = ScanCount;
      if (startSmoothScan_>-1)
        startScan = startSmoothScan_;
      if (stopSmoothScan_>-1)
        stopScan = stopSmoothScan_;
      for (int j=0; j<repeats; j++)
      {
        for (int i=startScan; i<stopScan; i++)  Value[i][3] = smoothMeanDataPoint(i, range);
        for (int i=startScan; i<stopScan; i++)  Value[i][2] = Value[i][3];
      }
      //scale the reduced fit values to the old raw intensities
      float rawMax = 0;
      float fitMax = 0;
      for (int i=startScan; i<stopScan; i++){
        if (Value[i][1]>rawMax) rawMax = Value[i][1];
      }
      for (int i=startScan; i<stopScan; i++){
        if (Value[i][2]>fitMax) fitMax = Value[i][2];
      }
      float scale = rawMax/fitMax;
      for (int i=startScan; i<stopScan; i++){
        Value[i][2] = Value[i][2]*scale;
      }
    }
    
    
    public void Smooth(float range, int repeats)
    {
      this.Smooth(range, repeats,true);
    }
    
    protected void copyRawData(){
      for (int i=0; i<ScanCount; i++)
      {
        Value[i][2] = Value[i][1];
      }

    }
    
    private float smoothMeanDataPoint(int dtIndex, float range){
      int lower = calcBoundIndex(dtIndex, range, false);
      int upper = calcBoundIndex(dtIndex, range, true);
      Vector<Double> values = new Vector<Double>();
      for (int i=lower; i!=upper; i++) values.add((double)Value[i][2]);
      return Calculator.mean(values);
    }
    
    private int calcBoundIndex(int dtIndx, float range, boolean posDirection){
      int boundIndex = dtIndx;
      if (posDirection){
        while (boundIndex<(ScanCount-1) && (Value[boundIndex][0] - Value[dtIndx][0])<range) ++boundIndex;
      }else{
        while (boundIndex>0 && (Value[dtIndx][0] - Value[boundIndex][0])<range) --boundIndex;
      }
      return boundIndex;
    }
    
    /**
     * Smoothes a rough chromatogram within a time range.
     * @param range
     *      number of seconds around given points
     * @param repeats
     *      number of smooth runs for each point
     */
    public void Smooth(float range, int repeats, boolean copyRawDataFirst)
    {
        float threshold;
        int				i, j;
        
        // =============================================================
        // Calculate the minimum y value and
        // store it into threshold:
        // =============================================================
        
        threshold = Value[0][1];
        for (i=0; i<ScanCount; i++){
            if(Value[i][1]<threshold) threshold = Value[i][1];
        }
        if (copyRawDataFirst) copyRawData();
        
        // =============================================================
        // Smooth each point around the given time range 
        // =============================================================
        
        for (j=0; j<repeats; j++)
        {
          int startScan = 0;
          int stopScan = ScanCount;
          if (startSmoothScan_>-1)
            startScan = startSmoothScan_;
          if (stopSmoothScan_>-1)
            stopScan = stopSmoothScan_;
          precalcPows_ = new float[Value.length]; 
          
          int preCalcStart = calcBoundIndex(startScan,range,false)-10;
          if (preCalcStart<0) preCalcStart = 0;
          int preCalcStop = calcBoundIndex(stopScan,range,true)+10;
          if (preCalcStop>ScanCount) preCalcStop =ScanCount;
          
          for (i=preCalcStart; i<preCalcStop; i++) precalcPows_[i] = (float)Math.pow(Value[i][2], 0.25); 
           for (i=startScan; i<stopScan; i++)  Value[i][3] = SmoothDataPoint(i, range, threshold);
           for (i=startScan; i<stopScan; i++)  Value[i][2] = Value[i][3];
        }
    }
    
    
    /**
     * smoothes rough spectrum at a specific point.
     * 
     * @param dtIndx
     *          Index of point to be smoothed
     * @param range
     *          number of seconds around given points
     * @param threshold
     *          minumum value to pass back
     * @return
     *          smoothed value
     */
    private float SmoothDataPoint(int dtIndx, float range, float threshold)
    {
        int			order = 4;
        int			lower, upper;
        float		val;
        
        // =============================================================
        // get boundary
        // =============================================================
        
        lower = calcBoundIndex(dtIndx,range,false);
        upper = calcBoundIndex(dtIndx,range,true);
        
        while(upper-lower<10)													
        {
            if (lower>0)                         --lower;
            if (upper<ScanCount-1)               ++upper;
            if( lower<=0 && upper>=ScanCount-1)  break;
        }
        
        // =============================================================
        // get filter value
        // =============================================================
        
        order = order < upper-lower-1 ? order : upper-lower-1;
        
        if(order < 1)       return Value[dtIndx][2];
        else                val = SavGolFilter(dtIndx, lower, upper, order);
        
        if(val > threshold) return val;
        else                return threshold;
    }
    
    
    /**
     * Does a Savitzky Golay Filter around a given point
     * 
     * @param dtIndx
     *          Index of point
     * @param lower
     *          lower range border
     * @param upper
     *          upper range border
     * @param order
     *          order of polynome
     * @return
     *          smoothed value
     */
    private float SavGolFilter(int dtIndx, int lower, int upper, int order)
    {
        float		mtrx[][];
        float		vec[];
        float		x = Value[dtIndx][0];
        int			indx[];
        float		sum;
        float adding = 0;
        int			i, j, k;
        
        indx = new int[order+1];
        vec = new float[order+1];
        mtrx = new float[order+1][order+1];
        
        // =================================================================
        // get "mtrx" 
        // =================================================================
        
        for (i=0; i<=order; i++) 
        {
            for (j=i; j<=order; j++) 
            {
                sum = 0;
                
                for (k=lower; k<=upper; k++) 
                {
//                    if (Value[k][2] > 1)
//                        sum += (float)(Math.pow(Value[k][0] - x, i+j)
//                                * Math.pow(Value[k][2], 0.25));
//                    else
//                        sum += (float)Math.pow(Value[k][0]-x, i+j);
 //                 precalcPows_[i] = (float)Math.pow(Value[i][2], 0.25); 

                  adding = 0f;

                  
//                  adding = (float)Math.pow(Value[k][0] - x, i+j);
                  for (int l=0; l<(i+j); l++){
                    if (l==0)
                      adding = (Value[k][0] - x);
                    else
                      adding = adding*(Value[k][0] - x);
                  }
                  if ((i+j) ==0) adding = 1f;
                  
                  
                  if (Value[k][2] > 1)
                    adding = adding * precalcPows_[k];
                  sum += adding;

                }
                mtrx[i][j] = sum;
                mtrx[j][i] = sum;
            }
        }
        
        // =================================================================
        // LU decomposition
        // =================================================================
        
        myLUDcmp(mtrx, order+1, indx);
        
        // =================================================================
        // get "vec"
        // =================================================================
        
        for (i=0; i<=order; i++) 
        {
            sum = 0;
            for (k=lower; k<=upper; k++) 
            {
                if (Value[k][2]>1)
                    sum += (float)(Math.pow(Value[k][0]-x, i) * Value[k][2]
                                                                         * Math.pow(Value[k][2], 0.25f));
                else
                    sum += (float)Math.pow(Value[k][0]-x, i)* Value[k][2];
            }
            vec[i] = sum;
        }
        
        // =================================================================
        // LU backsubstition
        // =================================================================
        
        myLUBksb(mtrx, order+1, indx, vec);
        x = vec[0];
        return x;
    }
    
    
    /**
     * performs LU decomposition
     * 
     * @param mtrx
     * @param order
     * @param indx
     */
    private void myLUDcmp(float mtrx[][], int order, int indx[])
    {
        float		big, dum, sum, temp;
        float		vv[];
        int			i, j, k, imax;
        float		d;
        
        vv = new float[order];
        
        d = 1.0F;
        imax = 0;
        for (i = 0; i < order; i++) 
        {
            big = 0.0f;
            for (j = 0; j<order; j++)
                if ((temp = Math.abs(mtrx[i][j])) > big) big = temp;
            vv[i] = 1.0F / big;
        } 
        
        for (j=0; j<order; j++) 
        {
            for (i=0; i<j; i++) 
            {
                sum = mtrx[i][j];
                for (k=0; k<i; k++) sum -= mtrx[i][k] * mtrx[k][j];
                mtrx[i][j] = sum;
            }
            big = 0.0f;
            for (i = j; i < order; i++) 
            {
                sum = mtrx[i][j];
                for (k=0; k<j; k++)
                    sum -= mtrx[i][k] * mtrx[k][j];
                mtrx[i][j] = sum;
                dum = vv[i] * Math.abs(sum);
                if (dum>=big) 
                {
                    big = dum;
                    imax = i;
                }
            }
            if (j!=imax) 
            {
                for (k=0; k<order; k++) 
                {
                    dum = mtrx[imax][k];
                    mtrx[imax][k] = mtrx[j][k];
                    mtrx[j][k] = dum;
                }
                d *= -1;
                vv[imax] = vv[j];
            }
            indx[j] = imax;
            if (mtrx[j][j]==0.0) mtrx[j][j] = 1E-20f;
            if (j!=order) 
            {
                dum = 1.0F / mtrx[j][j];
                for (i=j+1; i<order; i++) mtrx[i][j] *= dum;
            }
        }
    }
    

    /**
     * performs LU backsubstition
     * 
     * @param mtrx
     * @param order
     * @param indx
     * @param vec
     */
    private void myLUBksb(float mtrx[][], int order, int indx[], float vec[])
    {
        float sum;
        int ii, ip;
        int i, j;
        
        ii = -1;
        for (i=0; i<order; i++) 
        {
            ip = indx[i];
            sum = vec[ip];
            vec[ip] = vec[i];
            if (ii != -1)
                for (j=ii; j<i; j++) sum -= mtrx[i][j]*vec[j];
            else if (sum!=0) ii = i;
            vec[i] = sum;
        }
        for (i=order-1; i>=0; i--) 
        {
            sum = vec[i];
            for (j=i+1; j<order; j++) sum -= mtrx[i][j]*vec[j];
            vec[i] = sum/mtrx[i][i];
        }
    }
    /**  it could happen that the peak you are looking for in the profile is a little bit shifted in m/z direction
       therefore this method is called which takes the first peak that is OK in the m/z neighborhood
    */   
    public void findPeakInNeighbourhood(int position, int direction, int valleyMethod, int backgroundValue){
      this.FindPeak(position, direction, valleyMethod);
      this.GetBackground(backgroundValue);
      this.GetAreaAndTime();
      int loScan = position;
      if (LoValley<loScan)
        loScan=LoValley;
      int upScan = position;
      if (UpValley>upScan)  
        upScan = UpValley;
      float highestArea = 0f;
      int highestAreaPeakPosition = -1;
      // if the peak is not found the algorithm is looking on both sides for valid peaks
      while ((!Good)&&(loScan>0||upScan<(ScanCount-1))){
        loScan--;
        if (loScan>-1){
          this.FindPeak(loScan, -1, valleyMethod);
          this.GetBackground(backgroundValue);
          this.GetAreaAndTime();
          if (LoValley<loScan)
            loScan=LoValley;
          if (this.Area>highestArea){
            highestArea = this.Area;
            highestAreaPeakPosition = this.Peak;
          }
        }
        if (!Good){
          upScan++;
          if (upScan<(ScanCount)){
            this.FindPeak(upScan, 1, valleyMethod);
            this.GetBackground(backgroundValue);
            this.GetAreaAndTime();
            if (UpValley>upScan)  
              upScan = UpValley;
            if (this.Area>highestArea){
              highestArea = this.Area;
              highestAreaPeakPosition = this.Peak;
            }
          }
        }
      }
      if (!Good && highestArea>0){
        this.FindPeak(highestAreaPeakPosition, direction, valleyMethod);
        this.GetBackground(backgroundValue);
        this.GetAreaAndTime();
        ////this.Good = true;
      }
//      if (Good){
//        correctPeakPositionWithRawData();
//      }
    }
    
    
    private void correctPeakPositionWithRawData(){
      float intensity = Value[Peak][2];
      int newPeakPosition = -1;
      for (int i=LoValley;i<(UpValley+1);i++){
        if (Value[i][1]>intensity){
          intensity = Value[i][1];
          newPeakPosition = i;
        }
      }
      // if we have a small adjacent peak - do not change to this value
      if (newPeakPosition>=0 && newPeakPosition!=LoValley && newPeakPosition!=UpValley)
        Peak = newPeakPosition;
    }
    
    private boolean isNothingThere(int position){
      boolean isNothingThere = true;
      if (Value[position][2]>0)
        isNothingThere = false;
      if (isNothingThere && position>0 && Value[position-1][2]>0)
        isNothingThere = false;
      if (isNothingThere && position<(ScanCount-1) && Value[position+1][2]>0)
        isNothingThere = false;
      if (isNothingThere){
        Peak = position;
        LoValley = Peak-1;
        UpValley = Peak+1;
        if (UpValley>ScanCount-2){
          UpValley = ScanCount-2;
          LoValley = ScanCount-4;
        }
        if (LoValley<1){
          LoValley = 1;
          UpValley = 3;
        }
      }
      return isNothingThere;
    }

    
    /**
     * This method finds the peak at the "position" in the fitted spectrum.
     * If the "position" is a valley, then find the left peak 
     * (when "direction = -1") or the right one (when "direction = 1").
     * 
     * @param position
     *          where to start
     * @param direction
     *          where to go if start point is a valley
     * @param valleyMethod
     *          The original one (0) or Erik's one (1)
     */
    public void FindPeak(int position, int direction, int valleyMethod)
    {
      
      if (isNothingThere(position)) return;
      else anythingThere = true;
      this.startPosition_ = position; 
      int right, left;
        int i;
        
        // =============================================================
        // determine slope on the right hand side 
        // =============================================================
        
        i = 1;
        while(position+i<ScanCount && Value[position][2]==Value[position+i][2])
        {
            ++i;
        } 
        if (position+i>=ScanCount)                       right = 0;
        else if (Value[position][2]< Value[position+i][2]) right = 1;
        else                                               right = -1;
        
        // =============================================================
        // determine slope on the left hand side
        // =============================================================
        
        i = -1;
        while(position+i >= 0 && Value[position][2] == Value[position+i][2])
        {
            --i;
        } 
        if (position+i < 0)                               left = 0;
        else if (Value[position][2]<Value[position+i][2]) left = 1;
        else                                              left = -1;
        
        if((right==-1 && left==-1) || (right==0 && left==0)) 
        {
          Peak = position;
            GetValleys(valleyMethod);
            return;
        }
        else if (right==-1 && left==0)                       
        {
            Peak = 0;
            GetValleys(valleyMethod);
            return;
        }
        else if (right==0 && left==-1)
        {
            Peak = ScanCount-1;
            GetValleys(valleyMethod);
            return;
        }
        
        // =============================================================
        // determine "direction"
        // =============================================================
        
        if (right<=0 && left==1)      direction = -1;
        else if (right==1 && left<=0) direction = 1;
        
        // =============================================================
        // search for peak
        // =============================================================
//        float differential1 = 0;
//        float differential2 = 0;
//        if (valleyMethod == CgDefines.GreedySteepnessReductionMethod){
//          differential1 = Value[position-direction][2]-Value[position][2];
//          differential2 = Value[position][2]-Value[position+direction][2];
//        }
        
        while (position+direction >= 0 && position+direction < ScanCount
                && Value[position][2] <= Value[position+direction][2]) 
        {
            position += direction;
        }
        
        // =============================================================
        // return index for peak
        // =============================================================
        
        if (position+direction < 0)               Peak = 0;
        else if (position+direction >= ScanCount) Peak = ScanCount - 1;
        else                                      Peak = position;
        GetValleys(valleyMethod);
    }
    
    
    /**
     * determines the valley by selecting the appropriate method
     * @param method
     *          The original one (0) or Erik's one (1)
     */
    protected void GetValleys(int method)
    {
        if (method==CgDefines.StandardValleyMethod)
            GetValleysOriginal();
        else if (method==CgDefines.EnhancedValleyMethod) 
            GetValleysEnhanced(true);
        else
            GetValleysOriginal();
    }
    
    
    /**
     * This function finds the next valley from left (when "direction 
     * = -1") or right (when "direction = 1"), starting at "position" 
     * in the smoothed chromatogram.
     */
    protected void GetValleysOriginal()
    {
        int position;
        
        // =============================================================
        // Do the upwards direction:
        // =============================================================
        
        position = Peak;
        while(position<ScanCount-1)
        {
            if (Value[position+1][2]>Value[position][2]) break;
            position++;
        }
        UpValley = position;
        if (UpValley>ScanCount-2) UpValley = ScanCount-2;
        
        // =============================================================
        // Do the downwards direction:
        // =============================================================
        
        position = Peak;
        while(position>0)
        {
            if (Value[position-1][2]>Value[position][2]) break;
            position--;
        }
        LoValley = position;
        if (LoValley<1) LoValley = 1;
    }
    
    /**
     * This function is similar to the original one. It finds the next 
     * valley from left ("direction" = -1) or right ("direction" = 1), 
     * starting at "position" in the smoothed chromatogram.
     * It recognizes valleys only in case their level is smaller than 0.1
     * of the peak and the peak's amplitude is higher than 10xaverage.
     */
    protected void GetValleysEnhanced(boolean refinePeakPosition)
    {
        int		position;
        float	localPeak;
        
        localPeak = Value[Peak][2];
        
        // =============================================================
        // Do the upwards direction:
        // =============================================================
        
        position = Peak;
        while(position<ScanCount-1)
        {   
            if (Value[position][2]>(0.1 * localPeak) && localPeak>(10*m_average))
            {
                position++;
                continue;
            }
            if (Value[position+1][2]>Value[position][2]) break;
            position++;
        }
        UpValley = position;
        if (UpValley>ScanCount-2) UpValley = ScanCount-2;
        
        // =============================================================
        // Do the downwards direction:
        // =============================================================
        
        position = Peak;
        while(position>0)
        {
            if (Value[position][2]>(0.1f * localPeak) && localPeak>(10*m_average)) 
            {
                position--;
                continue;
            }
            if (Value[position-1][2]>Value[position][2]) break;
            position--;
        }
        // if (position==0) return;
        LoValley = position;
        if (LoValley<1) LoValley = 1;
        if (refinePeakPosition){
          this.refinePeakPosition();
          this.GetValleysEnhanced(false);
        }  
    }
    
    
    private void refinePeakPosition(){
      int position = Peak;
      float highestValue = Value[Peak][2];
      for (int i=LoValley; i!=UpValley;i++){
        if (Value[i][2]>highestValue){
          highestValue = Value[i][2];
          position = i;
        }
      }
      this.Peak = position;
    }
    
     /**
     * Calculates average and peak od the Chromatogram
     */
    public void GetMaximumAndAverage()
    {
        int			i;
        
        m_peakValue = 0;
        m_average = 0;
        
        for (i=0; i<ScanCount; i++)
        {
            if (Value[i][2]>m_peakValue) m_peakValue = Value[i][2];
            m_average += Value[i][2];
        }
        m_average /= ScanCount;
    }
    
    
    /**
     * Determines the background of the chromatogram
     * @param range
     *      # of scans right and left of the peak
     */
    public void GetBackground(int range)
    {
        float noise, background, ave;
        int count1, count2;
        int dnRange, upRange;
        int i;
        
        // =============================================================
        // Set the background scan range
        // =============================================================
        
        dnRange = LoValley - range;
        if (dnRange<0) dnRange = 0;
        upRange = UpValley + range;
        if (upRange>=ScanCount) upRange = ScanCount-1;
        
        // =============================================================
        // get noise level
        // We cannot be sure if  the peak is really surrounded by noise
        // or if there is possibly an additional peak.
        // Therefore the noise on both sides of the peak is considered!
        // If the one side of the range is more than 200 times higher in 
        // intensity just the the one range with the lower intensity is taken
        // into account 
        // =============================================================
        float downSum = 0;
        float upSum = 0;
        for (i = dnRange; i <= upRange; ++i)
        {
          if(i<LoValley)
            downSum+= (Value[i][1] + Value[i][2])/2;
          if(i>UpValley)
            upSum+= (Value[i][1] + Value[i][2])/2;
        }
        boolean takeJustDown = false;
        boolean takeJustUp = false;
        if (upSum/downSum>200)
          takeJustDown = true;
        if (downSum/upSum>200)
          takeJustUp = true;
        
        noise = 0;
        count1 = 0;
        for (i = dnRange; i <= upRange; ++i)
        {
            if (Value[i][1] > 0) 
            {
              int rightPosition = i;
              // at the down range seems to be a peak so take the position at the up range again
              if(i<LoValley&&takeJustUp){
                rightPosition = UpValley+i;
                if (rightPosition>upRange)
                  continue;
              }
              // at the up range seems to be a peak so take the position at the down range again
              if(i>UpValley&&takeJustDown){
                rightPosition = LoValley-(i-UpValley);
                if (rightPosition<dnRange)
                  continue;
              }
              noise += (Value[rightPosition][1] - Value[rightPosition][2])
                * (Value[rightPosition][1] - Value[rightPosition][2]);
                count1++;
            }
        }
        if(count1 > 0) noise = (float)Math.sqrt(noise / count1);
        else           noise = 1;
        // =============================================================
        // get background
        // =============================================================
        
        background = 0;
        ave = noise;
        while(Math.abs(background - ave)>(0.01f * noise)) 
        {
            background = ave;
            ave = 0;
            count1 = 0;
            count2 = 0;
            int count3 = 0;
            for (i=dnRange; i<=upRange; ++i)
            {
                if(i<LoValley || i>UpValley) 
                {
                  int rightPosition = i;
                  if(i<LoValley&&takeJustUp){
                    rightPosition = UpValley+i;
                    if (rightPosition>upRange)
                      continue;
                  }
                  // at the up range seems to be a peak so take the position at the down range again
                  if(i>UpValley&&takeJustDown){
                    rightPosition = LoValley-(i-UpValley);
                    if (rightPosition<dnRange)
                      continue;
                  }
                  
                  if (Value[rightPosition][1]>0 && Value[rightPosition][1]<(background + noise)) 
                    {
                        ave += Value[rightPosition][1];
                        count1++;
                    }
                    else if (Value[rightPosition][1] >= (background + noise)) count2++;
                  // this is necessary for profiles; the values are there very rare and if just high values
                  // exist, the background could reach enormous values
                  if (this.isProfile_ && Value[rightPosition][1] >= (background + 2*noise)){
                    count3++;
                  }
                }
            }
            
            if(count1>0)       ave /= count1;
            // this is necessary for profiles; the values are there very rare and if just high values
            // exist, the background could reach enormous values
            else if(count2>0&&(count2==count3)){
              ave = 2*noise;
              break;
            }else if (count2>0)ave = background + 0.2f * noise;
            else               ave = 0;
        }
        Background = ave;
    }
    
    
    public void getAreaRaw(){
      int i;
      float rawVal;
      float rawArea = 0;
      
      for (i = 0; i != (ScanCount-1); i++) 
      {
        rawVal = (Value[i+1][1] + Value[i][1]) / 2 - Background;
        if (rawVal<0) rawVal = 0;
          rawArea += rawVal * (Value[i+1][0] - Value[i][0]);          
      }
      Area = rawArea;
    }
    
    /**
     * Dinds the area and time of a peak. It returns 2 
     * if the peak passes certain tests, 1 if not.
     */
    public void GetAreaAndTime()
    {
        float rawArea, fitArea;
        float rawVal,  fitVal;
        float maxVal, ave, areaErr;
        int dnIndex, upIndex;
        int count;
        int i;
        this.highestIntensity_ = 0f;
        
        this.Good = true;
        
        // =============================================================
        // Get Maximum
        // =============================================================
        
        maxVal = 0;
        for (i = LoValley; i <= UpValley; ++i) 
        {
          // Juergen: this was previously calculated on the smoothed values, while the background
          //          on raw values; Mean smoothing reduced the intensity, and so good hits were
          //          removed; thus, I changed this to the raw max value (better comparabele to background)
          //if (maxVal<Value[i][2]) maxVal = Value[i][2];
          if (maxVal<Value[i][1]) maxVal = Value[i][1];
        }
        
        // =============================================================
        // Get the Peak Area
        // =============================================================
        
        rawArea = 0;
        fitArea = 0;
        areaErr = 0;
        for (i = LoValley; i <= UpValley; i++) 
        {
//          System.out.println("i: "+i);
          int addValue = 1;
          // high resolution profile raw data has values with no intensity in between,
          // therefore we must jump to the next intensity value
          if (this.isProfile_){
            while (Value[i+addValue][1]==0&&addValue!=5&&(i+addValue)<(ScanCount-1))
              addValue++;
          }
            rawVal = (Value[i+addValue][1] + Value[i][1]) / 2 - Background;
            if (rawVal<0) rawVal = 0;
            float value = rawVal * (Value[i+addValue][0] - Value[i][0]);
            rawArea += value;
            if (value>highestIntensity_)
              highestIntensity_ = value;
            
            fitVal = (Value[i+1][2] + Value[i][2]) / 2 - Background;
            if (fitVal<0) fitVal = 0;
            fitArea += fitVal * (Value[i+1][0] - Value[i][0]);
            
            areaErr += 0.5f * (rawVal - fitVal) * (rawVal - fitVal)
            * (Value[i+1][0] - Value[i][0]) 
            * (Value[i+1][0] - Value[i][0]);
        }
        
        Area = (fitArea + rawArea) / 2;
        
        // =============================================================
        // The Area calculates is bad in case that:
        // - the raw area is <= 0
        // - the fitted area is <= 0
        // - the calculated area is smaller than the difference between
        //   the raw and fitted area
        // - the maximum intensity is smaller than half of the backgnd
        // =============================================================
        
        AreaErr = (float)Math.sqrt(areaErr);
        
        
        /** Comment: In my opinion the old solution was wrong, if max value must be bigger than 2* the background it would never reach the double check on bad data */
        if(rawArea<=0 || fitArea<=0 || Area< AreaErr || maxVal < /****(Background)*/2 * Background) 
        {
            Good = false;
        }
        // =============================================================
        // find peakTime 
        // =============================================================
        
        PeakTime = Value[Peak][0];
        
        // find error in peakTime
        
        float threshold = (Value[Peak][2] + Background) / 2;
        dnIndex = Peak;
        while(dnIndex >= LoValley)
        {
            if (Value[dnIndex][2] <= threshold) break;
            dnIndex--;
        }
        
        upIndex = Peak;
        while(upIndex <= UpValley)
        {
            if (Value[upIndex][2] <= threshold) break;
            upIndex++;
        }
        
        if (upIndex<=dnIndex) PeakTimeAverage = 0;
        else                  PeakTimeAverage = (Value[upIndex][0] - Value[dnIndex][0]) / 2;
        
        /** Comment: here was standing maxVal > Background/2; nevertheless the peak would have
         * been already bad, since the previous check some lines above was that the peak should be 
         * 2 times as high; (see previous comment)
         */
        if (Area==0 || maxVal>=/****Background/2*/Background*2) return;
        
        // =============================================================
        // Double check on bad data:
        // =============================================================
        
        count = 0;
        for (i=dnIndex; i<=upIndex; i++)
        {
            if(Value[i][1]<Background) count++;
        }
        if(count>(upIndex-dnIndex+1)/4) 
        {
            count = 0;
            if (Value[dnIndex][2]<Value[upIndex][2]) ave = Value[dnIndex][2];
            else                                     ave = Value[upIndex][2];
            
            for (i=dnIndex; i<=upIndex; i++)
            {
                if (Value[i][1]>ave) count++;
            }
            /** I uncommented the +1 to give a higher chance to narrower peaks*/
            if (count<(upIndex-dnIndex/***+1*/)/2) Good = false;
        }
    }
    
    /**
     * Method to save the Chromatogram in a tab separated value file with given 
     * name. The first column contains the retention times, the second column 
     * the raw values and the third column the smoothed values.
     * 
     * @param fileName
     *        Contains path and file name of destination file.
     */
    public void Save(String fileName) throws CgException
    {
        int         i;
        
        try
        {
            File outputTextFile = new File(fileName);
            FileWriter textFileWriter = new FileWriter(outputTextFile);
            
            for (i=0; i<ScanCount; i++)
            {
                textFileWriter.write(Float.toString(Value[i][0]) + "\t");
                textFileWriter.write(Float.toString(Value[i][1]) + "\t");
                textFileWriter.write(Float.toString(Value[i][2]) + "\r\n");                            
            }
            textFileWriter.close();
        }
        catch (Exception ex)
        {
            throw new CgException("Unable to write file " + fileName) ;
        }
    }

    public float getHighestIntensity()
    {
      return highestIntensity_;
    }

    public void setHighestIntensity(float highestIntensity)
    {
      this.highestIntensity_ = highestIntensity;
    }

    public float getM_average()
    {
      return m_average;
    }

    public int getStartPosition()
    {
      return startPosition_;
    }

    public float getM_peakValue()
    {
      return m_peakValue;
    }

    
    
    
//  public void smoothingBorderCorrection(int nrOfScans){
//    if ((this.LoValley+nrOfScans)<this.Peak)
//      this.LoValley = this.LoValley+nrOfScans;
//    if ((this.UpValley-nrOfScans)>this.Peak)
//      this.UpValley = this.UpValley-nrOfScans;
//  }
    
    
    /**
     * interpolates the true measured values of a chromatogram with fixed time points which are used as input
     * @param retentionTimes the time points were the interpolation shall take place
     */
    public void doChromValueInterpolation(Hashtable<Integer,Float> retentionTimes){
      int amountOfScans = retentionTimes.size();
      float[][] newValue = new float[amountOfScans][4];
      int currentIndex = 0;
      for (int i=0;i!=amountOfScans;i++){
        float rt = (retentionTimes.get(new Integer(i))).floatValue();
        newValue[i][0] = rt;
        newValue[i][1] = interpolateWithNeighbors(rt, currentIndex);
        if (newValue[i][0]>Value[currentIndex][0]) currentIndex++;
        if (currentIndex>=ScanCount) currentIndex--;
      }      
      ScanCount = amountOfScans;
      Value = newValue;
    }
    
    /**
     * calculates an interpolation intensity value for a given time point
     * @param rt the retention time point where the interpolation shall take place
     * @param dtIndx the closest index in the non-interpolated chromatogram
     * @return the interpolated value
     */
    private float interpolateWithNeighbors(float rt, int dtIndx){
      float value = 0f;
      if (rt<=Value[0][0]){
        value = Value[0][1];
      } else if (rt>=Value[ScanCount-1][0]){
        value = Value[ScanCount-1][1];
      } else if (Value[dtIndx][0]>rt){
        int idx = dtIndx-1; 
        while (Value[idx][0]>rt) idx--;
        value = distWeightedValue(rt, idx, (idx+1));
      } else{
        int idx = dtIndx+1;
        while (Value[idx][0]<rt) idx++;
        value = distWeightedValue(rt, (idx-1), idx);
      }
      return value;
    }
    
    
    /**
     * calculates an interpolation value
     * @param rt the retention time where the interpolation shall take place
     * @param low the next lower index
     * @param up the next upper index
     * @return the interpolated value
     */
    private float distWeightedValue(float rt, int low, int up){    
      float dist = Value[up][0]-Value[low][0];
      float lowerPerc = (Value[up][0]-rt)/dist;
      float upperPerc = 1f-lowerPerc;
      return Value[low][1]*lowerPerc+Value[up][1]*upperPerc;
    }


}
