/*
 * Created on 24.06.2005
 * Erik Koerner, FH Joanneum
 * erik.koerner@fh-joanneum.at
 */
package at.tugraz.genome.maspectras.quantification;

import java.io.File;
import java.io.FileWriter;
import java.text.*;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.Vector;

/**
 * Core class for Chromatogram Peak Area analysis. Aggregates the scans read in
 * from an mzXML file and processes CgParameterSets. Implements the CgIAddScan
 * interface for the callbacks used by the CgMzXML class.
 * 
 * @author Erik Koerner
 * @version 25.06.2005
 */
public class CgAnalyzer extends Analyzer implements CgIAddScan 
{
  private String m_fileName; // The mzXml File Name

  private CgScanHeader m_header; // My Header

  private CgScan[] m_scans; // My Scans

  private int m_scanCount; // Actual Scan Count (used as index)

  /**
   * Reads in an mzXML file during construction.
   * 
   * @param mzXmlPath
   *          path and file name of the mzXML file to be read in.
   */
  public CgAnalyzer(String mzXmlPath, String fileType)
  {
    this(mzXmlPath, fileType, false);
  }

  /**
   * Reads in an mzXML file during construction.
   * 
   * @param mzXmlPath
   *          path and file name of the mzXML file to be read in.
   * @param includeMSMS
   *          include MSMS levels
   */
  public CgAnalyzer(String mzXmlPath, String fileType, boolean includeMSMS)
  {
    super();
    CgReader m_reader; // my MzXml Reader (I need it only once...

    m_fileName = mzXmlPath;
    m_scanCount = 0;
    m_chroma = new CgChromatogram[CgDefines.MaxCharge];
    this.numberOfCachedParts_ = 0;
    try {
      m_reader = null;
      if (fileType.equalsIgnoreCase("mzXML")) {
        m_reader = new CgMzXmlReader(this, includeMSMS);
/*        this.highestMz_ = ((CgMzXmlReader)m_reader).getHighestMz()/CgDefines.mzMultiplicationFactorForInt;
        this.lowestMz_ = ((CgMzXmlReader)m_reader).getLowestMz()/CgDefines.mzMultiplicationFactorForInt;*/
      }
      if (fileType.equalsIgnoreCase("mzData")) {
        System.out.println("Taking a mzData File");
        m_reader = new CgMzDataReader(this);
      }
      if (fileType.equalsIgnoreCase("RAW")) {
        System.out.println("Taking a XCalibur RAW File");
        m_reader = new CgXCaliburRawReader(this);
      }
      if (m_reader != null)
        m_reader.ReadFile(m_fileName);
      else
        throw new CgException("No readable file format.");
      /*
       * for (int i=0;i!=m_scans.length;i++){ System.out.println("ScanCount:
       * "+i); System.out.println("Subsans: "+m_scans[i].SubScans.size()); }
       */
    }
    catch (Exception ex) {
      System.out.println(ex.getMessage());
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @see conGenetics.CgIAddScan#AddScan(conGenetics.CgScan)
   */
  public void AddScan(CgScan sx) throws CgException
  {
    if (m_scans == null)
      throw new CgException("m_scans Array not allocated");
    if (m_scans.length == m_scanCount)
      return;
    m_scans[m_scanCount] = sx;
    m_scanCount++;
  }

  public void AddHeader(CgScanHeader hx) throws CgException
  {
    if (m_header != null)
      throw new CgException("Data Header already defined");
    m_header = hx;
    m_scans = new CgScan[hx.ScanCount];
    m_scanCount = 0;
  }

  public void setStartStopHeader(CgScanHeader hx) throws CgException
  {
    if (m_header == null)
      throw new CgException("No header defined");
    m_header.StartTime = hx.StartTime;
    m_header.EndTime = hx.EndTime;
  }


  public void SaveResults(String pathName) throws CgException
  {
    int i;
    CgProbe px;
    NumberFormat fmt, fma;

    try {
      for (i = 0; i < CgDefines.MaxCharge; i++) {
        if (m_chroma[i] != null)
          m_chroma[i].Save(pathName + "ChromaData_" + i + ".txt");
      }
    }
    catch (Exception ex) {
      throw new CgException(ex.getMessage());
    }

    try {
      fmt = new DecimalFormat("#0.00");
      fma = new DecimalFormat("#0");

      File outputTextFile = new File(pathName + "ResultData.txt");
      FileWriter textFileWriter = new FileWriter(outputTextFile);

      textFileWriter.write("Peptide: " + m_params.Peptide + "\r\n");
      textFileWriter.write("Total Area determined ..... "
          + fma.format(m_params.Area) + "\r\n");
      textFileWriter.write("\r\n");

      for (i = 0; i < m_params.ProbeCount(); i++) {
        px = m_params.Probe(i);
        textFileWriter.write("******** Scan=" + px.Scan + " Charge="
            + px.Charge + " ********\r\n");
        textFileWriter.write("\r\n");
        textFileWriter.write("m/z Value ................. " + fmt.format(px.Mz)
            + "\r\n");
        textFileWriter.write("Lower m/z Band ............ "
            + fmt.format(px.LowerMzBand) + "\r\n");
        textFileWriter.write("Upper m/z Band ............ "
            + fmt.format(px.UpperMzBand) + "\r\n");
        textFileWriter.write("\r\n");
        textFileWriter.write("Peak Lower Valley [min] ... "
            + fmt.format(px.LowerValley / 60) + "\r\n");
        textFileWriter.write("Peak Center [min] ......... "
            + fmt.format(px.Peak / 60) + "\r\n");
        textFileWriter.write("Peak Upper Valley [min] ... "
            + fmt.format(px.UpperValley / 60) + "\r\n");
        textFileWriter.write("\r\n");
        textFileWriter.write("Peak Amplitude (raw) ...... "
            + fma.format(px.PeakAmplitudeRaw) + "\r\n");
        textFileWriter.write("Peak Amplitude (fit) ...... "
            + fma.format(px.PeakAmplitudeFit) + "\r\n");
        textFileWriter.write("Background Amplitude ...... "
            + fma.format(px.Background) + "\r\n");
        if (px.AreaStatus == CgAreaStatus.Undefined) {
          textFileWriter.write("Area ...................... Undetermined\r\n");
        }
        if (px.AreaStatus == CgAreaStatus.OK) {
          textFileWriter.write("Area ...................... "
              + fma.format(px.Area) + "\r\n");
        }
        if (px.AreaStatus == CgAreaStatus.TooSmall) {
          textFileWriter.write("Area ...................... Too small\r\n");
        }
        if (px.AreaStatus == CgAreaStatus.Duplicate) {
          textFileWriter.write("Area ...................... Duplicate\r\n");
        }
        textFileWriter.write("\r\n");
      }
      textFileWriter.close();
    }
    catch (Exception ex) {
      throw new CgException("Unable to write results file ");
    }
  }


  public Collection<CgScan> getScans(SortedSet<Integer> scanNbrs)
  {

    class ScanHelper
    {
      CgScan scan_;

      public boolean written_ = false;

      public int level_;

      private ScanHelper parent_;

      public ScanHelper(CgScan scan)
      {
        scan_ = scan;
        level_ = scan.MsLevel;
      }

      public ScanHelper(CgScan scan, ScanHelper parent)
      {
        scan_ = scan;
        level_ = scan.MsLevel;
        if (parent.level_ < level_)
          parent_ = parent;
        else
          parent_ = parent.getParent(level_);
      }

      public ScanHelper getParent(int level)
      {
        if (level_ == level)
          return parent_;
        else
          return parent_.getParent(level);
      }

      public void write(Collection<CgScan> list)
      {
        if (!written_) {
          if (parent_ != null)
            parent_.write(list);
          list.add(scan_);
          written_ = true;
        }
      }
    }

    ScanHelper helper = null;
    Vector<CgScan> list = new Vector<CgScan>();

    if (scanNbrs.size() < 0)
      return null;
    Iterator<Integer> it = scanNbrs.iterator();
    int scanNbr = it.next();

    for (CgScan scan:m_scans) {
      if (scan != null) {
        helper = new ScanHelper(scan);
        if (scan.Num == scanNbr) {
          helper.write(list);
          if (it.hasNext())
            scanNbr = it.next();
          else
            break;
        }
        for (CgScan subscan:scan.getFullSubScans()) {
          if (subscan != null) {
            helper = new ScanHelper(subscan, helper);
            if (subscan.Num == scanNbr) {
              helper.write(list);
              if (it.hasNext())
                scanNbr = it.next();
              else
                break;
            }
          }
        }
      }
    }
    return list;
  }


  
  protected int findBaseScanIndex(int scanNumber)
  {
    int i;

    for (i = 0; i < m_scanCount; i++) {
      if (m_scans[i].Contains(scanNumber)){
        return i;
      }  
    }
    return 0;
  }
  
  protected void readTheChromatograms()throws CgException{
    int i;
    
    
    for (i = 0; i < CgDefines.MaxCharge; i++) {
      if (m_params.Mz[i] > 0) {
//        long time = System.currentTimeMillis();
        m_chroma[i] = this.readAChromatogram(m_params.Mz[i],m_params.LowerMzBand,m_params.UpperMzBand);
//        System.out.println("Smoothing Time: "+(System.currentTimeMillis()-time));
      } else {
        m_chroma[i] = null;
      }
    }
  }
  
  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand, float smoothRange, int smoothRepeats) throws CgException{
    float retTime, intensity;
    CgChromatogram cx = new CgChromatogram(m_scanCount); // make new Chromatogram
    cx.Mz = mz;
    cx.LowerMzBand = lowerMzBand;
    cx.UpperMzBand = upperMzBand;
    for (int j = 0; j < m_scanCount; j++) {
      retTime = m_scans[j].RetentionTime;
      intensity = m_scans[j].GetIntensity(cx.Mz - cx.LowerMzBand, cx.Mz
          + cx.UpperMzBand);
      cx.Value[j][0] = retTime;
      cx.Value[j][1] = intensity;
    }
//    System.out.println("Reading Time: "+(System.currentTimeMillis()-time));
//    time = System.currentTimeMillis();
    cx.Smooth(smoothRange, smoothRepeats);
    return cx;
  }
  
  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand, float smoothRange,int smoothRepeats, float startTime, float stopTime) throws CgException{
    return readAChromatogram(mz, lowerMzBand, upperMzBand,smoothRange,smoothRepeats);
  }
  
  public int getTimeShiftedScanNumber(int scanNumber, float timeShift){
    int baseScanIndex = this.findBaseScanIndex(scanNumber);
    float timeToLookAt = this.m_scans[baseScanIndex].RetentionTime+timeShift;
    int shiftedIndex = this.m_scans[baseScanIndex].Num;
    if (timeShift>0){
      CgScan lastScan = this.m_scans[baseScanIndex];
      for (int i=baseScanIndex;i!=this.m_scans.length;i++){
        System.out.println(this.m_scans[i].RetentionTime);
        if (timeToLookAt < this.m_scans[i].RetentionTime)
           return lastScan.Num;
        else
          lastScan = this.m_scans[i];
      }
    }else if (timeShift<0){
      for (int i=baseScanIndex; i!=-1; i--){
        if (timeToLookAt > this.m_scans[i].RetentionTime){
          return this.m_scans[i].Num;
        }
      }
    }
    return shiftedIndex;
  }

  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand,
      float upperMzBand) throws CgException
  {
    return this.readAChromatogram(mz, lowerMzBand, upperMzBand,0.5f,10);
  }


  public CgScan getLastBaseScan(){
    for (int i=(m_scanCount-1); i>-1; i--){
      if (m_scans[i]!=null&&m_scans[i].MsLevel==1)
        return m_scans[i];
    }
    return null;
  }
  
  public void addParentFileName(String fileName) throws CgException{
    //TODO: put your implementation here
  }
}
