package at.tugraz.genome.maspectras.chromaviewer;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.math.BigDecimal;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.border.Border;

import at.tugraz.genome.maspectras.msResultsAnalysis.vos.ViewerresolutionVO;
import at.tugraz.genome.maspectras.msResultsAnalysis.vos.ViewersettingVO;
import at.tugraz.genome.maspectras.quantification.CgProbe;
import at.tugraz.genome.maspectras.quantification.Probe3D;
import at.tugraz.genome.maspectras.utils.Calculator;
import at.tugraz.genome.viewer3d.MapViewer;

public class MSMapViewer extends MapViewer
{
  
  public static final int DISPLAY_TIME_SECONDS = 0;
  
  public static final int DISPLAY_TIME_MINUTES = 1;
    
  private int displayTime_ = DISPLAY_TIME_SECONDS;

  private static final long serialVersionUID = 6846466196675299985L;

  private Hashtable<Integer,Float> retentionTimes;
  
  protected String xAxisString = "time";
  protected String yAxisString = "intensity";
  protected String zAxisString = "m/z";
  
  protected String resolutionLabelXString = "t: ";
  protected String resolutionLabelZString = "m/z: ";
  
  protected String stretchXString = "Stretch time:";
  protected String stretchYString = "Stretch int.:";
  protected String stretchZString = "Stretch m/z:";
  
  protected JCheckBox showSelectedCheckBox;
  protected JCheckBox show2DPositionCheckBox;
  protected JCheckBox isDefaultCheckBox;
  
  private JLabel mzStartLabel_;
  private JLabel mzStopLabel_;
  private JTextField mzStartText_;
  private JTextField mzStopText_;
  
  protected JButton saveSettings_;
  
  private float startMz;
  private float stopMz;
  private int resolution;
  private float stepSize;
  
  private float start2DMz;
  private float stop2DMz;
  
  private float start2DTime;
  private float stop2DTime;
  
  private Vector brownProbes;
  private Vector redProbes;
  private Vector blueProbes;
  private Vector greenProbes;
  private Vector goldProbes;
  private Vector orangeProbes;
  private Vector violetProbes;
  private Vector turquoiseProbes;
  private Vector pinkProbes;
  private Vector bluePastelProbes;
  private Vector greenBlueProbes;
  private Vector greenPastelProbes;

  
  public final static Color COLOR_BROWN = new Color(219,132,30);
  public final static Color COLOR_RED = new Color(240,0,19);
  public final static Color COLOR_BLUE = new Color(40,97,255);
  public final static Color COLOR_GREEN = new Color(55,255,30);
  public final static Color COLOR_GOLD = new Color(206,196,26);
  public final static Color COLOR_ORANGE = new Color(255,160,43);
  public final static Color COLOR_VIOLET = new Color(153,0,250);
  public final static Color COLOR_TURQUOISE = new Color(21,240,240);
  public final static Color COLOR_PINK = new Color(255,195,215);
  public final static Color COLOR_BLUE_PASTEL = new Color(204,204,255);
  public final static Color COLOR_GREEN_BLUE = new Color(0,153,153);
  public final static Color COLOR_GREEN_PASTEL = new Color(153,255,153);

 
  public final static String CMD_SHOW_SELECTED = "showSelected";
  protected final static String CMD_SHOW_2D_POSITION = "show2DPosition";
  
  protected ActionListener theCallingWindow_;
  
  public final static String CMD_SAVE_SETTINGS = "saveSettings";
  public final static String CMD_3D_DEFUALT = "set3dDefault";
  
  protected final static Color DARK_YELLOW = new Color(100,100,000);
  
  protected final static Color DARK_RED = new Color(100,0,000);
  protected final static Color BRIGHT_RED = new Color(255,0,000);
  
  protected final static Color DARK_GREEN = new Color(000,100,000);
  protected final static Color BRIGHT_GREEN = new Color(000,255,000);
  
  protected BufferedImage yellowGradient_;
  protected BufferedImage brownGradient_;
  protected BufferedImage redGradient_;
  protected BufferedImage blueGradient_;
  protected BufferedImage greenGradient_;
  protected BufferedImage goldGradient_;
  protected BufferedImage orangeGradient_;
  protected BufferedImage violetGradient_;
  protected BufferedImage turquoiseGradient_;
  protected BufferedImage pinkGradient_;
  protected BufferedImage bluePastelGradient_;
  protected BufferedImage greenBlueGradient_;
  protected BufferedImage greenPastelGradient_;

  
  private boolean theShow2dWasOn = false;
  private boolean theShowSelectedWasOn = false;
  
  private float stretchXValue = 2f;
  private float stretchYValue = 1f;
  private float stretchZValue = 1f;
  private boolean lightValue = true;
  private boolean textureValue = true;
  private boolean selectedValue = false;
  private boolean positionValue = false;
  private boolean isDefaultValue = false;
  private int resolutionSize = 2;
  private String[] initialMzResolutions = null;
  private String[] initialTimeResolutions = null;
  private String[] initialDistanceResolutions = null;
  private Hashtable<Integer,Hashtable<Integer,Float[]>> ellipseBorders_ = null;
  /** TODO: this is a fixed value for the highest Resolution of the viewer, i.e. 0.01mDa; this should be possibly changed  in future*/
  private int highestResolution_ = 100000;
  private boolean msRangeChangeable_;
  private float[][] originalData_;
  
  public MSMapViewer(float[][] data, Hashtable<Integer,Float> retentionTimes, float startMz, float stopMz, int resolution, float stepSize){
    super(data);
    this.retentionTimes = retentionTimes;
    this.startMz = startMz;
    this.stopMz = stopMz;
    this.resolution = resolution;
    this.stepSize = stepSize;
    this.theCallingWindow_ = this;
    this.ellipseBorders_ = new Hashtable<Integer,Hashtable<Integer,Float[]>>();
    msRangeChangeable_ = false;
    resetPaintableProbes();
  }
  
  private void resetPaintableProbes(){
    brownProbes = new Vector();
    redProbes = new Vector();
    blueProbes = new Vector();
    greenProbes = new Vector();
    goldProbes = new Vector();
    orangeProbes = new Vector();
    violetProbes = new Vector();
    turquoiseProbes = new Vector();
    pinkProbes = new Vector();
    bluePastelProbes = new Vector();
    greenBlueProbes = new Vector();
    greenPastelProbes = new Vector();
  }
  
  public MSMapViewer(float[][] data, Hashtable<Integer,Float> retentionTimes, float startMz, float stopMz, int resolution, float stepSize, ActionListener theCallingWindow){
    this(data, retentionTimes, startMz, stopMz, resolution, stepSize);
    this.theCallingWindow_ = theCallingWindow;
  }
  
  public MSMapViewer(float[][] data, Hashtable<Integer,Float> retentionTimes, float startMz, float stopMz, int resolution, float stepSize, ActionListener theCallingWindow, int displayTime, boolean msRangeChangeable){
    this(data, retentionTimes, startMz, stopMz, resolution, stepSize,theCallingWindow);
    this.displayTime_ = displayTime;
    msRangeChangeable_ = msRangeChangeable;
    if (msRangeChangeable_) originalData_ = data;
  }
  
  protected float getXValueRange(){
    return this.retentionTimes.get(super.data[0].length-1)-this.retentionTimes.get(0);
    
  }
  
  protected float getXValueForPosition(int position){
    return this.retentionTimes.get(position);
  }
  
  protected String getXAxisLabel(){
    return this.xAxisString;
  }
  
  protected String getZAxisLabel(){
    return this.zAxisString;
  }
  
  protected String getYAxisLabel(){
    return this.yAxisString;
  }
  
  protected void createResolutionFields(){    
    JLabel label;
    JTextField text;
    
    boolean onlyOne = false;
    int size = this.resolutionSize;
    
    if (size==1)
      onlyOne = true;
    for (int i=0; i!=size;i++){
      String resolutionString = "Res. "+(i+1);
      if (i==0&&!onlyOne)
        resolutionString+=" (far)";
      if (i==(size-1)&&!onlyOne)
        resolutionString+=" (close)";
      label = new JLabel(resolutionString);
      this.setStandardJLabelProperties(label);
      this.resolutionLabels.add(label); 
      
      label = new JLabel(this.getResolutionLabelXString());
      this.setStandardJLabelProperties(label);
      this.resolutionXLabels.add(label);
      

      label = new JLabel(this.getResolutionLabelZString());
      this.setStandardJLabelProperties(label);
      this.resolutionZLabels.add(label);
      
      if (i!=(size-1)){
        label = new JLabel("d: ");
        this.setStandardJLabelProperties(label);
        this.resolutionDistLabels.add(label);
      }
      
      text = new JTextField(5);
      if (i==0&&this.initialTimeResolutions==null){
        text.setText("10");
      }
      if (i==1&&this.initialTimeResolutions==null)
        text.setText("5");
      if (initialTimeResolutions!=null){
        text.setText(String.valueOf(this.initialTimeResolutions[i]));
      }
      
      this.setStandardJTextProperties(text);
      this.resolutionXText.add(text);
      
      text = new JTextField(5);
      this.setStandardJTextProperties(text);
      this.resolutionZText.add(text);
      if (i==0&&this.initialMzResolutions==null){
        text.setText("0.1");
        if (this.stepSize/20<0.1f)
          text.setText(String.valueOf(this.stepSize/20));
      }  
      if (i==1&&this.initialMzResolutions==null){
        text.setText("0.02");    
        if (this.stepSize/100<0.1f)
          text.setText(String.valueOf(this.stepSize/100));
      }
      if (initialMzResolutions!=null){
        text.setText(String.valueOf(this.initialMzResolutions[i]));
      }
      
      if (i!=(size-1)){
        text = new JTextField(5);
        this.setStandardJTextProperties(text);
        this.resolutionDistText.add(text); 
        if (i==0&&this.initialDistanceResolutions==null)
          text.setText("0.8");
        if (this.initialDistanceResolutions!=null){
          text.setText(String.valueOf(this.initialDistanceResolutions[i]));
        }
      }  

    }
  }
  
  protected void createResolutionErrorMessage(String text, int position){
    this.rightMenuBar.setPreferredSize(new Dimension(170,50));
    JLabel label = new JLabel(text);
    this.setStandardJLabelProperties(label);
    label.setForeground(Color.red);
    rightMenuBar.add(label,new GridBagConstraints(2, position+getRightMenubarYOffset(), 1, 1, 0.0, 0.0
        ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));    
  }
  
  protected String getResolutionLabelDistString()
  {
    return this.resolutionLabelDistString;
  }


  protected String getResolutionLabelXString()
  {
    return this.resolutionLabelXString;
  }


  protected String getResolutionLabelZString()
  {
    return this.resolutionLabelZString;
  }
  
  protected String getStretchXLabel(){
    return this.stretchXString;
  }
  
  protected String getStretchYLabel(){
    return this.stretchYString;
  }
  
  protected String getStretchZLabel(){
    return this.stretchZString;
  }
  
  protected float getLowestZValue(){
    if (this.msRangeChangeable_)
      return new Float(mzStartText_.getText());
    return this.startMz;
  }
  
  protected float getHighestZValue(){
    if (this.msRangeChangeable_)
      return new Float(mzStopText_.getText());
    return this.stopMz;
  }
  
  protected float getLowestXValue(){
    return this.retentionTimes.get(0);
  }
  
  protected float getHighestXValue(){
    return this.retentionTimes.get(super.data[0].length-1);
  }
  
  protected float getZValueForPosition(int position){
    return this.getLowestZValue()+(position)/Float.valueOf(String.valueOf(this.resolution));
  }

  
  protected boolean checkResolutionInput(){
    boolean allCorrect = true;
    JTextField field;
    float lastXValue = 0;
    float thisXValue;
    float lastZValue = 0;
    float thisZValue;
    float lastDistValue = 0;
    float thisDistValue;

    topMenuBar.setPreferredSize(new Dimension(100,30));
    Component[] comp = topMenuBar.getComponents();
    for (int i=0;i!=comp.length;i++){
      if (comp[i] instanceof JLabel){
        String message = ((JLabel)comp[i]).getText();
        if (message.equalsIgnoreCase(">0!")||message.equalsIgnoreCase("Float!")||message.equalsIgnoreCase("Input!"))
          topMenuBar.remove(comp[i]);
      }
    }
    
    boolean checkValue = this.checkTopMenuFields();
    if (!checkValue) allCorrect = false;

    
    this.rightMenuBar.removeAll();
    this.rightMenuBar.setPreferredSize(new Dimension(95,50));
    this.setResolutionMenuPositions();

    
    for (int i=0; i!=this.resolutionLabels.size();i++){
      field = this.resolutionXText.get(i);
      if (field.getText()!=null&&field.getText().length()>0){
        try{
          thisXValue = Float.parseFloat(field.getText());
          if (i!=0&&thisXValue>lastXValue){
            allCorrect = false;
            this.createResolutionErrorMessage("closer -> smaller!", i*4+1);
          }
          lastXValue=thisXValue;  
        }catch(NumberFormatException nfx){
          allCorrect = false;
          this.createResolutionErrorMessage("Enter float!", i*4+1);
        }
      }else{
        allCorrect = false;
        this.createResolutionErrorMessage("Enter float!", i*4+1);
      }
      field = this.resolutionZText.get(i);
      if (field.getText()!=null&&field.getText().length()>0){
        try{
          thisZValue = Float.parseFloat(field.getText());
          if (i!=0&&thisZValue>lastZValue){
            allCorrect = false;
            this.createResolutionErrorMessage("closer -> smaller!", i*4+2);
          }
          lastZValue=thisZValue;  
        }catch(NumberFormatException nfx){
          allCorrect = false;
          this.createResolutionErrorMessage("Enter float!", i*4+2);
        }
      }else{
        allCorrect = false;
        this.createResolutionErrorMessage("Enter input!", i*4+2);
      }
      if (i!=this.resolutionLabels.size()-1){
        field = this.resolutionDistText.get(i);
        if (field.getText()!=null&&field.getText().length()>0){
          try{
            thisDistValue = Float.parseFloat(field.getText());
            if (i!=0&&thisDistValue>=lastDistValue){
              allCorrect = false;
              this.createResolutionErrorMessage("closer -> smaller!", i*4+3);
            }
            lastDistValue=thisDistValue;  
          }catch(NumberFormatException nfx){
            allCorrect = false;
            this.createResolutionErrorMessage("Enter float!", i*4+3);
          }
        }else{
          allCorrect = false;
          this.createResolutionErrorMessage("Enter input!", i*4+3);
        }
      }
    }
    if (this.msRangeChangeable_){
      this.setStartStopTextPositionsInRightMenuBar();
      boolean correct = this.checkMzStartStopInput();
      if (!correct) allCorrect = false;
    }
    if (this.msRangeChangeable_&&allCorrect){
      changeDataArrayToDisplay();
    }
    
    this.validate();
    this.rightMenuBar.repaint();
    this.topMenuBar.repaint();
    return allCorrect;
  }
  
  protected int getAmountForZResolution(int fieldIndex){
    float res = Float.parseFloat(this.resolutionZText.get(fieldIndex).getText());
    int amountOfLines = Integer.parseInt((new StringTokenizer(String.valueOf(res*this.resolution),".")).nextToken());
    if (amountOfLines<1) amountOfLines = 1;
    return amountOfLines;
  }
  
  protected int getAmountForXResolution(int fieldIndex){
    float res = Float.parseFloat(this.resolutionXText.get(fieldIndex).getText());
    //float currentTime = 0;
    int amountOfLines = 0;
    float timeSpace = retentionTimes.get(retentionTimes.size()-1)-retentionTimes.get(0);
    int requiredTimePoints =  (int)MapViewer.roundFloat(timeSpace/res,0,BigDecimal.ROUND_UP);
    amountOfLines = 1;
    if (requiredTimePoints<retentionTimes.size())
      amountOfLines = retentionTimes.size()/requiredTimePoints;
//    while (currentTime<res){
//      currentTime = this.retentionTimes.get(amountOfLines);
//      amountOfLines++;
//    }
//    System.out.println("Amount of lines time: "+amountOfLines+";"+retentionTimes.size()+";"+timeSpace);
    return amountOfLines;
  }
  
  protected float getInitialXStretchValue(){
    return this.stretchXValue;
  }
  protected float getInitialYStretchValue(){
    return this.stretchYValue;
  }
  protected float getInitialZStretchValue(){
    return this.stretchZValue;
  }

  protected boolean getInitialLightValue(){
    return this.lightValue;
  }
  
  protected boolean getInitialTextureValue(){
    return this.textureValue;
  }
  
  private boolean getInitialSelectedValue(){
    return this.selectedValue;
  }
  
  private boolean getInitialPositionValue(){
    return this.positionValue;
  }
  
  private boolean getInitialIsDefaultValue(){
    return this.isDefaultValue;
  }
  
  protected void createTopMenu()
  {
    Font toolBarFont=new Font("Dialog",Font.PLAIN,11);
    
    super.createTopMenu();
    super.topMenuBar.remove(this.updateResolutionLevel1);
    
    
    showSelectedCheckBox = new JCheckBox("Selected") {
      private static final long serialVersionUID = -8763724081406892698L;

      public Border getBorder() {
        return BorderFactory.createMatteBorder(0,0,0,2,new Color(50,50,50));
      }         
   };
   showSelectedCheckBox.setBorder(BorderFactory.createLineBorder(Color.white,1));
   showSelectedCheckBox.setBackground(new Color(50,50,50));
   showSelectedCheckBox.setOpaque(true);
   showSelectedCheckBox.setForeground(Color.white);
   showSelectedCheckBox.setIcon(new ImageIcon(MapViewer.class.getResource("/at/tugraz/genome/viewer3d/images/GenesisCheckBox-3.gif")));
   showSelectedCheckBox.setSelectedIcon(new ImageIcon(MapViewer.class.getResource("/at/tugraz/genome/viewer3d/images/GenesisCheckBox-Selected-3.gif")));
   showSelectedCheckBox.setFont(toolBarFont);
   showSelectedCheckBox.setFocusPainted(false);
   showSelectedCheckBox.setSelected(this.selectedValue);
   showSelectedCheckBox.setActionCommand(MSMapViewer.CMD_SHOW_SELECTED);
   showSelectedCheckBox.addActionListener(this);

   topMenuBar.add(this.showSelectedCheckBox,new GridBagConstraints(8, 1, 1, 1, 0.0, 0.0
       ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 6, 0, 0), 0, 0));
   
   show2DPositionCheckBox = new JCheckBox("2D-Position") {
    private static final long serialVersionUID = 2749902064207930180L;

    public Border getBorder() {
       return BorderFactory.createMatteBorder(0,0,0,2,new Color(50,50,50));
     }         
  };
  show2DPositionCheckBox.setBorder(BorderFactory.createLineBorder(Color.white,1));
  show2DPositionCheckBox.setBackground(new Color(50,50,50));
  show2DPositionCheckBox.setOpaque(true);
  show2DPositionCheckBox.setForeground(Color.white);
  show2DPositionCheckBox.setIcon(new ImageIcon(MapViewer.class.getResource("/at/tugraz/genome/viewer3d/images/GenesisCheckBox-3.gif")));
  show2DPositionCheckBox.setSelectedIcon(new ImageIcon(MapViewer.class.getResource("/at/tugraz/genome/viewer3d/images/GenesisCheckBox-Selected-3.gif")));
  show2DPositionCheckBox.setFont(toolBarFont);
  show2DPositionCheckBox.setFocusPainted(false);
  show2DPositionCheckBox.setSelected(this.positionValue);
  show2DPositionCheckBox.setActionCommand(MSMapViewer.CMD_SHOW_2D_POSITION);
  show2DPositionCheckBox.addActionListener(this);

  topMenuBar.add(this.show2DPositionCheckBox,new GridBagConstraints(9, 1, 1, 1, 0.0, 0.0
      ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 6, 0, 0), 0, 0));

  isDefaultCheckBox = new JCheckBox("3D is default view") {
    private static final long serialVersionUID = 2749902064207930180L;

    public Border getBorder() {
       return BorderFactory.createMatteBorder(0,0,0,2,new Color(50,50,50));
     }         
  };
  
  isDefaultCheckBox.setBorder(BorderFactory.createLineBorder(Color.white,1));
  isDefaultCheckBox.setBackground(new Color(50,50,50));
  isDefaultCheckBox.setOpaque(true);
  isDefaultCheckBox.setForeground(Color.white);
  isDefaultCheckBox.setIcon(new ImageIcon(MapViewer.class.getResource("/at/tugraz/genome/viewer3d/images/GenesisCheckBox-3.gif")));
  isDefaultCheckBox.setSelectedIcon(new ImageIcon(MapViewer.class.getResource("/at/tugraz/genome/viewer3d/images/GenesisCheckBox-Selected-3.gif")));
  isDefaultCheckBox.setFont(toolBarFont);
  isDefaultCheckBox.setFocusPainted(false);
  isDefaultCheckBox.setSelected(this.isDefaultValue);
  isDefaultCheckBox.setActionCommand(MSMapViewer.CMD_3D_DEFUALT);

  topMenuBar.add(this.isDefaultCheckBox,new GridBagConstraints(10, 1, 1, 1, 0.0, 0.0
      ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 6, 0, 0), 0, 0));
   
   topMenuBar.add(this.updateResolutionLevel1,new GridBagConstraints(11, 1, 1, 1, 0.0, 0.0
       ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 6, 0, 0), 0, 0));
   
    
   this.saveSettings_ = new JButton("Save Settings");
   this.saveSettings_.setActionCommand(MSMapViewer.CMD_SAVE_SETTINGS);
   this.saveSettings_.addActionListener(this.theCallingWindow_);
   this.setStandardJButtonProperties(saveSettings_);
   saveSettings_.setMargin(new Insets(1,8,1,8));
   topMenuBar.add(this.saveSettings_,new GridBagConstraints(12, 1, 1, 1, 0.0, 0.0
     ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 35, 0, 0), 0, 0));
   
   if (!this.equals(this.theCallingWindow_)){
     this.showLightCheckBox.addActionListener(this.theCallingWindow_);
     this.showTextureCheckBox.addActionListener(this.theCallingWindow_);
     this.showSelectedCheckBox.addActionListener(this.theCallingWindow_);
     this.show2DPositionCheckBox.addActionListener(this.theCallingWindow_);
     this.isDefaultCheckBox.addActionListener(this.theCallingWindow_);
     this.updateResolutionLevel1.addActionListener(this.theCallingWindow_);
   }
  }
  
  protected void createRightMenu()
  {
    super.createRightMenu();
    if (this.msRangeChangeable_){
      mzStartLabel_ = new JLabel("start: ");
      setStandardJLabelProperties(mzStartLabel_);
      mzStopLabel_ = new JLabel("stop: ");
      setStandardJLabelProperties(mzStopLabel_);
      mzStartText_ = new JTextField(5);
      mzStartText_.setText((new StringTokenizer(String.valueOf(MSMapViewer.roundFloat(startMz, 0, BigDecimal.ROUND_HALF_UP)),".")).nextToken());
      this.setStandardJTextProperties(mzStartText_);
      mzStopText_ = new JTextField(5);
      this.setStandardJTextProperties(mzStopText_);
      setStartStopTextPositionsInRightMenuBar();
      mzStopText_.setText((new StringTokenizer(String.valueOf(MSMapViewer.roundFloat(stopMz, 0, BigDecimal.ROUND_HALF_UP)),".")).nextToken());
    }
    if (!this.equals(this.theCallingWindow_)){
      super.updateResolutionLevel2.addActionListener(this.theCallingWindow_);
    }
    
  }
  
  public void setCurrent2DMzRange(float startMz, float stopMz){
    this.start2DMz = startMz;
    this.stop2DMz = stopMz;
  }
  
  public void setCurrent2DTimeRange(float startTime, float stopTime){
    this.start2DTime = startTime;
    this.stop2DTime = stopTime;
  }
  
  public void setPaintableProbes(Vector allProbes){
    resetPaintableProbes();
    if (allProbes.size()==2){
      this.redProbes = new Vector((Vector)allProbes.get(0));
      this.greenProbes = new Vector((Vector)allProbes.get(1));
    }
  }
  
  public void setMulticolorProbes(Hashtable<Color,Vector<CgProbe>> probes){
    for (Color key: probes.keySet()){
      if (key.equals(COLOR_BROWN)) this.brownProbes = probes.get(key);
      else if (key.equals(COLOR_RED)) this.redProbes = probes.get(key);
      else if (key.equals(COLOR_BLUE)) this.blueProbes = probes.get(key);
      else if (key.equals(COLOR_GREEN)) this.greenProbes = probes.get(key);
      else if (key.equals(COLOR_GOLD)) this.goldProbes = probes.get(key);
      else if (key.equals(COLOR_ORANGE)) this.orangeProbes = probes.get(key);
      else if (key.equals(COLOR_VIOLET)) this.violetProbes = probes.get(key);
      else if (key.equals(COLOR_TURQUOISE)) this.turquoiseProbes = probes.get(key);
      else if (key.equals(COLOR_PINK)) this.pinkProbes = probes.get(key);
      else if (key.equals(COLOR_BLUE_PASTEL)) this.bluePastelProbes = probes.get(key);
      else if (key.equals(COLOR_GREEN_BLUE)) this.greenBlueProbes = probes.get(key);
      else if (key.equals(COLOR_GREEN_PASTEL)) this.bluePastelProbes = probes.get(key);
    }
  }
  
/*  protected void setInitialViewPoint(TransformGroup tg, Point3d basis) {
    Transform3D t3d = new Transform3D();
    t3d.lookAt(new Point3d(1.4, 0.0, 3.5), basis, new Vector3d(0, 1, 0));
    t3d.invert();
    tg.setTransform(t3d);
  } */ 
  
  public void actionPerformed(ActionEvent event)
  {
    super.actionPerformed(event);
    String command = event.getActionCommand();
    if (command.equalsIgnoreCase(MSMapViewer.CMD_SHOW_2D_POSITION)){
      this.repaintColors();
    }
    if (command.equalsIgnoreCase(MSMapViewer.CMD_SHOW_SELECTED)){
      this.repaintColors();
    }
  }
  
  protected BufferedImage getColorGradient(float xValue, float yValue, float zValue){
    BufferedImage gradient = super.getColorGradient(xValue, yValue, zValue);
    if (this.show2DPositionCheckBox.isSelected()&&((zValue>=this.start2DMz&&zValue<=this.stop2DMz)||(xValue>=this.start2DTime&&xValue<=this.stop2DTime)))
      gradient = this.getYellowGradientImage();
    if (this.showSelectedCheckBox.isSelected()){
      if (this.brownProbes.size()>0 && this.checkInBrownProbe(xValue, yValue, zValue))
        gradient = this.getBrownGradientImage();
      else if (redProbes.size()>0 && this.checkInRedProbe(xValue, yValue, zValue))
        gradient = this.getRedGradientImage();
      else if (blueProbes.size()>0 && this.checkInBlueProbe(xValue, yValue, zValue))
        gradient = this.getBlueGradientImage();
      else if (greenProbes.size()>0 &&this.checkInGreenProbe(xValue, yValue, zValue))
        gradient = this.getGreenGradientImage();
      else if (goldProbes.size()>0 &&this.checkInGoldProbe(xValue, yValue, zValue))
        gradient = this.getGoldGradientImage();
      else if (orangeProbes.size()>0 &&this.checkInOrangeProbe(xValue, yValue, zValue))
        gradient = this.getOrangeGradientImage();
      else if (violetProbes.size()>0 &&this.checkInVioletProbe(xValue, yValue, zValue))
        gradient = this.getVioletGradientImage();
      else if (turquoiseProbes.size()>0 &&this.checkInTurquoiseProbe(xValue, yValue, zValue))
        gradient = this.getTurquoiseGradientImage();
      else if (pinkProbes.size()>0 &&this.checkInPinkProbe(xValue, yValue, zValue))
        gradient = this.getPinkGradientImage();
      else if (bluePastelProbes.size()>0 &&this.checkInBluePastelProbe(xValue, yValue, zValue))
        gradient = this.getBluePastelGradientImage();
      else if (greenBlueProbes.size()>0 &&this.checkInGreenBlueProbe(xValue, yValue, zValue))
        gradient = this.getGreenBlueGradientImage();
      else if (greenPastelProbes.size()>0 &&this.checkInGreenPastelProbe(xValue, yValue, zValue))
        gradient = this.getGreenPastelGradientImage();
      
    }
    return gradient;
  }
  
  private boolean checkInBrownProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.brownProbes.size();i++){
      if (this.checkProbe((CgProbe)brownProbes.get(i),i,xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }
  
  private boolean checkInRedProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.redProbes.size();i++){
      if (this.checkProbe((CgProbe)redProbes.get(i),i+this.brownProbes.size(),xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }
  
  private boolean checkInBlueProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.blueProbes.size();i++){
      if (this.checkProbe((CgProbe)blueProbes.get(i),i+brownProbes.size()+redProbes.size(),xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }
  
  private boolean checkInGreenProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.greenProbes.size();i++){
      if (this.checkProbe((CgProbe)greenProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size(),xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }

  private boolean checkInGoldProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.goldProbes.size();i++){
      if (this.checkProbe((CgProbe)goldProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size()+greenProbes.size(),xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }
  
  private boolean checkInOrangeProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.orangeProbes.size();i++){
      if (this.checkProbe((CgProbe)orangeProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size()+greenProbes.size()+goldProbes.size(),xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }

  private boolean checkInVioletProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.violetProbes.size();i++){
      if (this.checkProbe((CgProbe)violetProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size()+greenProbes.size()+goldProbes.size()+orangeProbes.size(),xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }

  private boolean checkInTurquoiseProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.turquoiseProbes.size();i++){
      if (this.checkProbe((CgProbe)turquoiseProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size()+greenProbes.size()+goldProbes.size()+orangeProbes.size()+violetProbes.size(),
          xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }
  
  private boolean checkInPinkProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.pinkProbes.size();i++){
      if (this.checkProbe((CgProbe)pinkProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size()+greenProbes.size()+goldProbes.size()+orangeProbes.size()+violetProbes.size()+turquoiseProbes.size(),
          xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }

  private boolean checkInBluePastelProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.bluePastelProbes.size();i++){
      if (this.checkProbe((CgProbe)bluePastelProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size()+greenProbes.size()+goldProbes.size()+orangeProbes.size()+violetProbes.size()+turquoiseProbes.size()+pinkProbes.size(),
          xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }

  private boolean checkInGreenBlueProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.greenBlueProbes.size();i++){
      if (this.checkProbe((CgProbe)greenBlueProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size()+greenProbes.size()+goldProbes.size()+orangeProbes.size()+violetProbes.size()+turquoiseProbes.size()+pinkProbes.size()+bluePastelProbes.size(),
          xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }

  private boolean checkInGreenPastelProbe(float xValue, float yValue, float zValue){
    boolean isInProbe = false;
    for (int i=0; i!=this.greenBlueProbes.size();i++){
      if (this.checkProbe((CgProbe)greenBlueProbes.get(i),i+brownProbes.size()+redProbes.size()+blueProbes.size()+greenProbes.size()+goldProbes.size()+orangeProbes.size()+violetProbes.size()+turquoiseProbes.size()+pinkProbes.size()+bluePastelProbes.size()+greenBlueProbes.size(),
          xValue, yValue,zValue)){
        isInProbe = true;
        break;
      }
    }
    return isInProbe;
  }

  
  private boolean checkProbe(CgProbe probe, int probePosition, float xValue, float yValue, float zValue){
    boolean returnValue = false;
    if (probe instanceof Probe3D){
      Probe3D probe3D = (Probe3D) probe;
      if ((probe3D.getEllipseMzPosition()-probe3D.getEllipseMzStretch())<=zValue&&zValue<=(probe3D.getEllipseMzPosition()+probe3D.getEllipseMzStretch())&&
          (probe3D.getEllipseTimePosition()-probe3D.getEllipseTimeStretch())<=xValue&&xValue<=(probe3D.getEllipseTimePosition()+probe3D.getEllipseTimeStretch())){
        Hashtable<Integer,Float[]> ellipseBorders = new Hashtable<Integer,Float[]>();
        if (this.ellipseBorders_.containsKey(probePosition))
          ellipseBorders = this.ellipseBorders_.get(probePosition);
        Integer borderIdentifer = (int)(zValue*(float)this.highestResolution_);
        if (!ellipseBorders.containsKey(borderIdentifer)){
          Float[] calcBorders = Calculator.calculateEllipseXBorderValues(zValue,probe3D.getEllipseTimePosition(),
              probe3D.getEllipseMzPosition(),probe3D.getEllipseTimeStretch(),probe3D.getEllipseMzStretch());
          if (probe3D.getLowerHardRtLimit()>=0 && probe3D.getLowerHardRtLimit()>calcBorders[0]) calcBorders[0] = probe3D.getLowerHardRtLimit();
          if (probe3D.getUpperHardRtLimit()>=0 && probe3D.getUpperHardRtLimit()<calcBorders[1]) calcBorders[1] = probe3D.getUpperHardRtLimit();
          ellipseBorders.put(borderIdentifer, calcBorders);
        }
        Float[] timeBorders = ellipseBorders.get(borderIdentifer);
        this.ellipseBorders_.put(probePosition, ellipseBorders);
        if (timeBorders[0]<=xValue&&xValue<=timeBorders[1]) returnValue=true;
      }  
    }else{
      if (xValue>=probe.LowerValley&&xValue<=probe.UpperValley&&
          zValue>=(probe.Mz-probe.LowerMzBand)&&zValue<=(probe.Mz+probe.UpperMzBand)) returnValue=true;
    }
    return returnValue;
  }
  
  private BufferedImage getYellowGradientImage(){
    if (this.yellowGradient_==null)
      this.yellowGradient_ = MSMapViewer.createYellowGradientImage(this.getDefaultNrOfGradientPixels());
    return this.yellowGradient_;
  }
  
  private BufferedImage getBrownGradientImage(){
    if (this.brownGradient_==null)
      this.brownGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_BROWN,MSMapViewer.COLOR_BROWN);
    return this.brownGradient_;
  }
  
  private BufferedImage getRedGradientImage(){
    if (this.redGradient_==null)
      this.redGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.DARK_RED,MSMapViewer.BRIGHT_RED);
    return this.redGradient_;
  }

  private BufferedImage getBlueGradientImage(){
    if (this.blueGradient_==null)
      this.blueGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_BLUE,MSMapViewer.COLOR_BLUE);
    return this.blueGradient_;
  }

  private BufferedImage getGreenGradientImage(){
    if (this.greenGradient_==null)
      this.greenGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.DARK_GREEN,MSMapViewer.BRIGHT_GREEN);
    return this.greenGradient_;
  }

  private BufferedImage getGoldGradientImage(){
    if (this.goldGradient_==null)
      this.goldGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_GOLD,MSMapViewer.COLOR_GOLD);
    return this.goldGradient_;
  }

  private BufferedImage getOrangeGradientImage(){
    if (this.orangeGradient_==null)
      this.orangeGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_ORANGE,MSMapViewer.COLOR_ORANGE);
    return this.orangeGradient_;
  }

  private BufferedImage getVioletGradientImage(){
    if (this.violetGradient_==null)
      this.violetGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_VIOLET,MSMapViewer.COLOR_VIOLET);
    return this.violetGradient_;
  }

  private BufferedImage getTurquoiseGradientImage(){
    if (this.turquoiseGradient_==null)
      this.turquoiseGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_TURQUOISE,MSMapViewer.COLOR_TURQUOISE);
    return this.turquoiseGradient_;
  }

  private BufferedImage getPinkGradientImage(){
    if (this.pinkGradient_==null)
      this.pinkGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_PINK,MSMapViewer.COLOR_PINK);
    return this.pinkGradient_;
  }

  private BufferedImage getBluePastelGradientImage(){
    if (this.bluePastelGradient_==null)
      this.bluePastelGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_BLUE_PASTEL,MSMapViewer.COLOR_BLUE_PASTEL);
    return this.bluePastelGradient_;
  }

  private BufferedImage getGreenBlueGradientImage(){
    if (this.greenBlueGradient_==null)
      this.greenBlueGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_GREEN_BLUE,MSMapViewer.COLOR_GREEN_BLUE);
    return this.greenBlueGradient_;
  }

  private BufferedImage getGreenPastelGradientImage(){
    if (this.greenPastelGradient_==null)
      this.greenPastelGradient_ = MSMapViewer.createGradientImage(this.getDefaultNrOfGradientPixels(),MSMapViewer.COLOR_GREEN_PASTEL,MSMapViewer.COLOR_GREEN_PASTEL);
    return this.greenPastelGradient_;
  }

  
  protected static BufferedImage createYellowGradientImage(int sizeX) {
    BufferedImage gradient = new BufferedImage(sizeX, 1, BufferedImage.TYPE_3BYTE_BGR);
    Graphics2D graphics = gradient.createGraphics();
    GradientPaint gp1 = new GradientPaint(0, 0, MSMapViewer.DARK_YELLOW, sizeX-1, 0, MapViewer.DEFAULT_COLOR2);
    graphics.setPaint(gp1);
    graphics.drawRect(0, 0, (sizeX-1), 1);
    graphics.dispose();    
    return gradient;
  }
  
  protected static BufferedImage createGradientImage(int sizeX, Color startColor, Color stopColor) {
    BufferedImage gradient = new BufferedImage(sizeX, 1, BufferedImage.TYPE_3BYTE_BGR);
    Graphics2D graphics = gradient.createGraphics();
    GradientPaint gp1 = new GradientPaint(0, 0, startColor, sizeX-1, 0, stopColor);
    graphics.setPaint(gp1);
    graphics.drawRect(0, 0, (sizeX-1), 1);
    graphics.dispose();    
    return gradient;
  }
  
  
  public void repaintColors(){
    long time = System.currentTimeMillis();
    this.ellipseBorders_ = new Hashtable<Integer,Hashtable<Integer,Float[]>>();
    if (this.theShow2dWasOn||this.show2DPositionCheckBox.isSelected()||
        (this.theShowSelectedWasOn&&(!this.showSelectedCheckBox.isSelected()))||((!this.theShowSelectedWasOn)&&this.showSelectedCheckBox.isSelected())){
      for (int k=this.resolutionLabels.size()-1; k!=-1;k--){
        int resZ = this.getAmountForZResolution(k);
        int resX = this.getAmountForXResolution(k);
        int amountZPositions = data.length/resZ;
        if (data.length%resZ!=0)
          amountZPositions++; 
        int amountXPositions = data[0].length/resX;
        if (data[0].length%resX!=0)
          amountXPositions++; 

        float[][] displayData = null; 
        if (data.length>0)
          displayData = new float[amountZPositions][amountXPositions];
        int j = -1;
        int m = -1;
        for (int i = 0; i != data.length; i++) {
          if (i % resZ == 0) {
            j++;
          }
          m = -1;
          for (int l=0;l!=data[0].length;l++){
            if (l % resX == 0){
              m++;
            }
            if (data[i][l]!=Float.NaN)
              displayData[j][m] += data[i][l];
          }
        }
        float[] colors = new float[displayData.length*displayData[0].length*3];
        for (int i=0; i!=(displayData.length); i++){
          for (j=0;j!=displayData[i].length;j++){
            float[] color = this.getColorForCoordinate(this.getXValueForPosition(j*resX),displayData[i][j],this.getZValueForPosition(i*resZ));//new float[3];

            colors[(i*displayData[0].length+j)*3] = color[0];
            colors[(i*displayData[0].length+j)*3+1] = color[1];
            colors[(i*displayData[0].length+j)*3+2] = color[2];
          }
        }
        super.stripArrays_[k].setColors(0,colors);
      }
    }
    if (this.show2DPositionCheckBox.isSelected()){
      this.theShow2dWasOn = true;
    }else{
      this.theShow2dWasOn = false;
    }
    if (this.showSelectedCheckBox.isSelected()){
      this.theShowSelectedWasOn = true;
    }else{
      this.theShowSelectedWasOn = false;
    }
    this.validate();
    this.repaint();
  }

  public void setTheShowSelectedWasOn(boolean theShowSelectedWasOn)
  {
    this.theShowSelectedWasOn = theShowSelectedWasOn;
  }
  
  private boolean checkTopMenuFields(){
    boolean allCorrect = true;
    boolean checkValue = this.checkTopMenuInput(this.stretchXField, 1);
    if (!checkValue) allCorrect = false;
    checkValue = this.checkTopMenuInput(this.stretchYField, 3);
    if (!checkValue) allCorrect = false;
    checkValue = this.checkTopMenuInput(this.stretchZField, 5);
    if (!checkValue) allCorrect = false;
    return allCorrect;
  }
  
  public ViewersettingVO getViewerSettings(){
    ViewersettingVO settingsVO = null;
    boolean allOk = true;
    if (!this.checkResolutionInput()) allOk=false;
    if (!this.checkTopMenuFields()) allOk=false;
    if (allOk){
      Vector<ViewerresolutionVO> resolutionSettings = new Vector<ViewerresolutionVO>();
      Double timeValue;
      Double mzValue;
      Double distance;
      for (int i=0;i!=super.resolutionXText.size();i++){
        timeValue = new Double(super.resolutionXText.get(i).getText());
        mzValue = new Double (super.resolutionZText.get(i).getText());
        if (i==(super.resolutionXText.size()-1)){
          distance = null;
        }else{
          distance = new Double(super.resolutionDistText.get(i).getText());
        }
        ViewerresolutionVO resolutionVO = new ViewerresolutionVO(null,new Integer(i+1),timeValue,mzValue,distance);
        resolutionSettings.add(resolutionVO);
      }
      Double timeStretch = new Double(super.stretchXField.getText());
      Double intensityStretch = new Double(super.stretchYField.getText());
      Double mzStretch = new Double (super.stretchZField.getText());
      Boolean showLight = new Boolean(super.showLightCheckBox.isSelected());
      Boolean showTexture = new Boolean(super.showTextureCheckBox.isSelected());
      Boolean showSelected = new Boolean(this.showSelectedCheckBox.isSelected());
      Boolean positionSelected = new Boolean(this.show2DPositionCheckBox.isSelected());
      Boolean use3dAsDefault = new Boolean(this.isDefaultCheckBox.isSelected());
      settingsVO = new ViewersettingVO(null,null,timeStretch, intensityStretch, mzStretch, showLight, showTexture, 
          showSelected, positionSelected, use3dAsDefault);
      settingsVO.setViewerresolutions(resolutionSettings);
    }  
    super.validate();
    super.rightMenuBar.repaint();
    super.topMenuBar.repaint();
    return settingsVO;
  }
  
  public void setViewerSettings(ViewersettingVO settingVO){
    if (settingVO!=null){
      this.stretchXValue = Float.parseFloat(String.valueOf(settingVO.getTimeStretch()));
      this.stretchYValue = Float.parseFloat(String.valueOf(settingVO.getIntensityStretch()));
      this.stretchZValue = Float.parseFloat(String.valueOf(settingVO.getMzStretch()));
      this.lightValue = settingVO.getLightSelected().booleanValue();
      this.textureValue = settingVO.getTextureSelected().booleanValue();
      this.selectedValue = settingVO.getSelectedSelected().booleanValue();
      this.positionValue = settingVO.getPositionSelected().booleanValue();
      this.isDefaultValue = settingVO.getThreeDDefault().booleanValue();
      Iterator resVOs = settingVO.getViewerresolutions().iterator();
      int resSize = settingVO.getViewerresolutions().size();
      this.resolutionSize = resSize;
      this.initialTimeResolutions = new String[resSize];
      this.initialMzResolutions = new String[resSize];
      this.initialDistanceResolutions = new String[resSize];
      while (resVOs.hasNext()){
        ViewerresolutionVO resVO = (ViewerresolutionVO)resVOs.next();
        int index = resVO.getResolutionLevel().intValue()-1;
        this.initialTimeResolutions[index] = resVO.getTimeResolution().toString();
        this.initialMzResolutions[index] = resVO.getMzResolution().toString();
        if (index!=(resSize-1))
          this.initialDistanceResolutions[index] = resVO.getDistance().toString();
      }
    }  
  }
  
  public void setViewerSettings(boolean lightValue, boolean textureValue,String mzResolution, String timeResolution){
    this.lightValue = lightValue;
    this.textureValue = textureValue;
    this.initialTimeResolutions = new String[1];
    this.initialMzResolutions = new String[1];
    this.initialDistanceResolutions = new String[1];
    this.resolutionSize = 1;
    this.initialTimeResolutions[0] = timeResolution;
    this.initialMzResolutions[0] = mzResolution;
  }
  
  public void setViewerSettings(boolean lightValue, boolean textureValue, boolean selectedValue,String mzResolution, String timeResolution){
    this.setViewerSettings(lightValue, textureValue, mzResolution, timeResolution);
    this.selectedValue = selectedValue;
  }
  
  public void removeSaveSettings(){
    this.topMenuBar.remove(this.show2DPositionCheckBox);
    this.topMenuBar.remove(this.showSelectedCheckBox);
    this.topMenuBar.remove(this.saveSettings_);
    this.topMenuBar.validate();
    this.topMenuBar.repaint();
  }
  
  public void removeSaveLipidomicsSettings(){
    this.topMenuBar.remove(this.isDefaultCheckBox);
    this.topMenuBar.remove(this.saveSettings_);
    this.topMenuBar.validate();
    this.topMenuBar.repaint();
  }
  
  public void destroyViewer(){
    System.out.println("I am finalizing");
    super.destroyViewer();
    if (this.greenGradient_!=null&&this.greenGradient_.getGraphics()!=null){
      this.greenGradient_.flush();
      this.greenGradient_.getGraphics().dispose();
    }  
    this.greenGradient_ = null;
    this.initialDistanceResolutions = null;
    this.initialMzResolutions = null;
    this.initialTimeResolutions = null;
    if (this.isDefaultCheckBox!=null&&this.isDefaultCheckBox.getGraphics()!=null)
      this.isDefaultCheckBox.getGraphics().dispose();
    this.isDefaultCheckBox = null;
    if (this.redGradient_!=null&&this.redGradient_.getGraphics()!=null){
      this.redGradient_.flush();
      this.redGradient_.getGraphics().dispose();
    }  
    this.redGradient_ = null;
    this.retentionTimes = null;
    this.rightMenuBar = null;
    if (this.saveSettings_.getGraphics()!=null)
      this.saveSettings_.getGraphics().dispose();
    this.saveSettings_ = null;

    if (this.show2DPositionCheckBox.getGraphics()!=null)
      this.show2DPositionCheckBox.getGraphics().dispose();
    this.show2DPositionCheckBox = null;
    if (this.showSelectedCheckBox.getGraphics()!=null)
      this.showSelectedCheckBox.getGraphics().dispose();
    this.showSelectedCheckBox = null;
    if (this.mzStartLabel_!=null && this.mzStartLabel_.getGraphics()!=null)
      mzStartLabel_.getGraphics().dispose();
    mzStartLabel_ = null;
    if (this.mzStopLabel_!=null && this.mzStopLabel_.getGraphics()!=null)
      mzStopLabel_.getGraphics().dispose();
    mzStopLabel_ = null;
    if (this.mzStartText_!=null && this.mzStartText_.getGraphics()!=null)
      mzStartText_.getGraphics().dispose();
    mzStartText_ = null;
    if (this.mzStopText_!=null && this.mzStopText_.getGraphics()!=null)
      mzStopText_.getGraphics().dispose();
    mzStopText_ = null;
    originalData_ = null;
    if (brownProbes!=null) brownProbes.removeAllElements();
    brownProbes = null;
    if (redProbes!=null) redProbes.removeAllElements();
    redProbes = null;
    if (blueProbes!=null) blueProbes.removeAllElements();
    blueProbes = null;
    if (greenProbes!=null) greenProbes.removeAllElements();
    greenProbes = null;
    if (goldProbes!=null) goldProbes.removeAllElements();
    goldProbes = null;
    if (orangeProbes!=null) orangeProbes.removeAllElements();
    orangeProbes = null;
    if (violetProbes!=null) violetProbes.removeAllElements();
    violetProbes = null;
    if (turquoiseProbes!=null) turquoiseProbes.removeAllElements();
    turquoiseProbes = null;
    if (pinkProbes!=null) pinkProbes.removeAllElements();
    pinkProbes = null;
    if (bluePastelProbes!=null) bluePastelProbes.removeAllElements();
    bluePastelProbes = null;
    if (greenBlueProbes!=null) greenBlueProbes.removeAllElements();
    greenBlueProbes = null;
    if (greenPastelProbes!=null) greenPastelProbes.removeAllElements();
    greenPastelProbes = null;

    this.theCallingWindow_ = null;
    if (this.yellowGradient_!=null&&this.yellowGradient_.getGraphics()!=null){
      this.yellowGradient_.flush();
      this.yellowGradient_.getGraphics().dispose();
    }
    this.yellowGradient_ = null;
    this.xAxisString = null;
    this.yAxisString = null;
    this.zAxisString = null;
    this.resolutionLabelXString = null;
    this.resolutionLabelZString = null;
    this.stretchXString = null;
    this.stretchYString = null;
    this.stretchZString = null;
    this.ellipseBorders_ = null;
    try {finalize();}catch (Throwable e) {}
  }
  
  protected void finalizeThisObject(){
  }
  
  protected float getXLabelCorrectionFactor(){
    if (this.displayTime_ == DISPLAY_TIME_MINUTES)
      return 60f;
    else
      return 1f;
  }
  
  protected int getRightMenubarYOffset(){
    if (this.msRangeChangeable_) return 2;
    else return 0;
  }
  
  private void  setStartStopTextPositionsInRightMenuBar(){
    rightMenuBar.add(this.mzStartLabel_,new GridBagConstraints(0, 0, 1, 1, 0.0, 0.0
      ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
    rightMenuBar.add(this.mzStartText_,new GridBagConstraints(1, 0, 1, 1, 0.0, 0.0
      ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
    rightMenuBar.add(this.mzStopLabel_,new GridBagConstraints(0, 1, 1, 1, 0.0, 0.0
        ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
    rightMenuBar.add(this.mzStopText_,new GridBagConstraints(1, 1, 1, 1, 0.0, 0.0
        ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
  }
  
  private boolean checkMzStartStopInput(){
    boolean allCorrect = true;
    if (mzStartText_.getText()!=null&&mzStartText_.getText().length()>0){
      try{
        float value = Float.parseFloat(mzStartText_.getText());
        if (value<startMz){
          mzStartText_.setText((new StringTokenizer(String.valueOf(MSMapViewer.roundFloat(startMz, 0, BigDecimal.ROUND_HALF_UP)),".")).nextToken());
        }
      }catch(NumberFormatException nfx){
        allCorrect = false;
        this.createResolutionErrorMessage("Enter float!", 0-this.getRightMenubarYOffset());
      }
    }else{
      allCorrect = false;
      this.createResolutionErrorMessage("Enter float!", 0-this.getRightMenubarYOffset());
    }
    if (mzStopText_.getText()!=null&&mzStopText_.getText().length()>0){
      try{
        float value = Float.parseFloat(mzStopText_.getText());
        if (value>stopMz){
          mzStopText_.setText((new StringTokenizer(String.valueOf(MSMapViewer.roundFloat(stopMz, 0, BigDecimal.ROUND_HALF_UP)),".")).nextToken());
        }
      }catch(NumberFormatException nfx){
        allCorrect = false;
        this.createResolutionErrorMessage("Enter float!", 1-this.getRightMenubarYOffset());
      }
    }else{
      allCorrect = false;
      this.createResolutionErrorMessage("Enter float!", 1-this.getRightMenubarYOffset());
    }
    return allCorrect;
  }
  
  private void changeDataArrayToDisplay(){
    System.out.println("resolution: "+resolution);
    int startMzInt = Math.round(this.startMz*(float)resolution);
    int requestedStartMzInt = Math.round(Float.parseFloat(mzStartText_.getText())*(float)resolution);
    int linesToSkip = requestedStartMzInt-startMzInt;
//    System.out.println(startMzInt+";"+requestedStartMzInt+";"+linesToSkip);
    int stopMzInt = Math.round(this.stopMz*(float)resolution);
    int requestedStopMzInt = Math.round(Float.parseFloat(mzStopText_.getText())*(float)resolution);
    int stopLineOffset = stopMzInt-requestedStopMzInt;
//    System.out.println(stopMzInt+";"+requestedStopMzInt+";"+stopLine);
    data = new float[originalData_.length-linesToSkip-stopLineOffset][originalData_[0].length];
    for (int i=linesToSkip;i!=(stopMzInt-startMzInt-stopLineOffset);i++){
      data[i-linesToSkip] = originalData_[i];
    }
  }
}
