package at.tugraz.genome.dbutilities;

import java.math.BigInteger;
import java.net.InetAddress;
import java.sql.Connection;
import java.sql.SQLException;

/**
 * <p>
 * This class is a unique ID generator. To improve databases performance the
 * length was reduced to a minimum necessary.
 * </p>
 * 
 * <p>
 * The 16 characters long printable unique ID is a base64 representation of a 12
 * bytes long ID obtained by concatenating:
 * </p>
 * 
 * 1) The 1 less significant byte of the machine IP address. This limits the
 * size of the cluster (or subnet) to 256 machines. <br>
 * 
 * 2) The 4 bytes memory address of the generator class instance. This limits
 * the machine memory space to 4GB. <br>
 * 
 * 3) The less significant 5 bytes of the current time in milliseconds. This 
 * limits the lifetime of the ID to 35 years. <br>
 * 
 * 4) A 2 byte synchronized counter to refine possible conflicts within the same
 * millisecond. This will refine the time precision at 15 nanoseconds. <br>
 * 
 * <p>
 * The uniqueness of the ID has two dimensions: space and time.
 * </p>
 * 
 * <p>
 * Items 1) and 2) will assures the uniqueness in the network/machine space.
 * Since it does not change during the lifetime of the generator it will be
 * computed only once at the construction time. 
 * </p>
 * 
 * <p>
 * Items 3) and 4) assures the uniqueness in time.
 * </p>
 * 
 * <p>
 * The class can easily be converted to a singleton or stateless session EJB
 * </p>
 * 
 * @author Adrian *
 * @created 23. Mai 2003
 * @version 1 
 */

public class TSSGUIDPKGenerator implements PKGenerator {

	// singleton
	private static TSSGUIDPKGenerator instance;

	private byte[] bytes12;

	/**
	 * Description of the Field
	 */
	public short loopCounter = 0;

	public static TSSGUIDPKGenerator getInstance() {
		if (instance == null) {
			return new TSSGUIDPKGenerator();
		} else {
			return instance;
		}
	}

	/**
	 * * Creates new TSSGUIDPKGenerator 
	 */
	private TSSGUIDPKGenerator() {
		try {
			// Prepare a buffer to hold the 12 bytes for the unique binary ID
			bytes12 = new byte[12];
			// Get the local IP address as a byte array and copy the less
			// significant byte.
			byte[] bytesIP = InetAddress.getLocalHost().getAddress();
			bytes12[0] = bytesIP[3];
			// Get the memory address for this object and copy the 4 bytes.
			int memAdr = System.identityHashCode(this);
			byte[] bytes4 = new byte[4];
			toFixSizeByteArray(new BigInteger(String.valueOf(memAdr)), bytes4);
			bytes12[1] = bytes4[0];
			bytes12[2] = bytes4[1];
			bytes12[3] = bytes4[2];
			bytes12[4] = bytes4[3];
			// At this point the first 5 bytes are set for the liftime of this
			// object
		} catch (Exception ignore) {
		}
	}

	/**
	 * Gets the primaryKey attribute of the TSSGUIDPKGenerator object. FIXXME,
	 * this takes the hashCode of the String, returned by the generateID method!
	 * Change to string primKeys !!!
	 * 
	 * @return The primaryKey value as Long
	 */
	public Long getPrimaryKey(String s) {
		return getLongPrimaryKey();
	}

	public Long getLongPrimaryKey() {
		return new Long(generateID().hashCode());
	}

	/**
	 * Gets the primaryKey attribute of the TSSGUIDPKGenerator object
	 * 
	 * @param s
	 *            Description of the Parameter
	 * @return The primaryKey value as String
	 */
	public String getStringPrimaryKey() {
		return generateID();
	}

	/**
	 * * Returns a 16 printable characters unique ID. * The ID is base64
	 * encoded 
	 * 
	 * @return Description of the Return Value
	 */
	public String generateBase64ID() {
		// Get the current time and ignore the most 3 significant bytes. Copy
		// the remaining 5 bytes.
		long timeNow = System.currentTimeMillis() & 0xFFFFFFFFFFL;
		byte[] bytes5 = new byte[5];
		toFixSizeByteArray(new BigInteger(String.valueOf(timeNow)), bytes5);
		bytes12[5] = bytes5[0];
		bytes12[6] = bytes5[1];
		bytes12[7] = bytes5[2];
		bytes12[8] = bytes5[3];
		bytes12[9] = bytes5[4];

		// Get the current counter reading and copy the 2 bytes
		short counter = getLoopCounter();
		byte[] bytes2 = new byte[2];
		toFixSizeByteArray(new BigInteger(String.valueOf(counter)), bytes2);
		bytes12[10] = bytes2[0];
		bytes12[11] = bytes2[1];

		// Encode the information in base64 and return the unique ID
		return new String(Base64.encode(bytes12));
	}

	/**
	 * Returns a 16 printable characters unique ID. Takes the base 64 encoding
	 * and replace '+' with '$' and '/' with '_' so the resulted string can be
	 * used to generate database table names and LDAP names.
	 * 
	 * @return Description of the Return Value
	 */
	public String generateID() {
		// Return the unique ID
		return generateBase64ID().replace('+', '$').replace('/', '_');
	}

	/**
	 * Get the counter value as a signed short
	 * 
	 * @return The loopCounter value
	 */
	private synchronized short getLoopCounter() {
		return loopCounter++;
	}

	/**
	 * This method transforms Java BigInteger type into a fix size byte array
	 * containing the two's-complement representation of the integer. The byte
	 * array will be in big-endian byte-order: the most significant byte is in
	 * the zeroth element. If the destination array is shorter then the
	 * BigInteger.toByteArray(), the the less significant bytes will be copy
	 * only. If the destination array is longer then the
	 * BigInteger.toByteArray(), destination will be left padded with zeros.
	 * 
	 * @param bigInt
	 *            Description of the Parameter
	 * @param destination
	 *            Description of the Parameter
	 */
	private static void toFixSizeByteArray(BigInteger bigInt, byte[] destination) {
		// Prepare the destination
		for (int i = 0; i < destination.length; i++) {
			destination[i] = 0x00;
		}
		// Convert the BigInt to a byte array
		byte[] source = bigInt.toByteArray();
		// Copy only the fix size length
		if (source.length <= destination.length) {
			for (int i = 0; i < source.length; i++) {
				destination[destination.length - source.length + i] = source[i];
			}
		} else {
			for (int i = 0; i < destination.length; i++) {
				destination[i] = source[source.length - destination.length + i];
			}
		}
	}

	/**
	 * Returns the primary key for the given table for an ORACLE database.
	 * 
	 * @return The primary key. A unique number, or NULL if the primary key is
	 *         not available.
	 */

	public String toString() {
		return "TSSGUIDPKGenerator";
	}

public void setPkValuesAtBeginning(Connection con, String tableName) throws SQLException{
  
}

}
