/*
 * Created on 23.06.2005
 * Erik Koerner, FH Joanneum
 * erik.koerner@fh-joanneum.at
 */
package at.tugraz.genome.maspectras.quantification;

import java.util.*;
import java.math.BigDecimal;

import at.tugraz.genome.maspectras.utils.Calculator;

/**
 * Representing an m/z Level 1 Scan.
 *
 * @author koerner
 */
public class CgScan
{
	public int		Num               = 0;     // Scan number in mzXML file
	public int		MsLevel           = 0;     // The scan's level (here always 1)
	public int		PeaksCount        = 0;     // Number of peaks
	public float	RetentionTime     = 0;     // The first peak's retention time
	public float	LowMz             = 0;     // from m/z
	public float	HighMz            = 0;     // to   m/z
	public float	BasePeakMz        = 0;     //
	public float	BasePeakIntensity = 0;     //
	public float	TotIonCurrent     = 0;     //

	public int		Precision         = 0;     // number of bits (32) for float
	public String	ByteOrder         = "";    // "network" normally
	public String	PairOrder         = "";    // what comes first (time-m/z)

	public float               Scan[][];       // our values
	public ArrayList<Integer>  SubScans;       // indices of subscans

  /** the polarity of the scan - CgDefines.POLARITY_NO is set when the mzXML file stores no polarity*/
  private int polarity_ = CgDefines.POLARITY_NO;

  /** dummy MS1 scans are set when there are MSn scans before MS1 scans*/
  private boolean dummyScan_; 

    private ArrayList<CgScan>   fullSubScans;

    /**
     * constructor initializing the required objects/arrays
     * @param peaksCount the amount of peaks
     */
    public CgScan(int peaksCount){
	      PeaksCount = peaksCount;
	      Scan = new float[peaksCount][2];
	      SubScans = new ArrayList<Integer>();
      fullSubScans = new ArrayList<CgScan>();
      dummyScan_ = false;
    }

    /**
     * 
     * @param peaksCount the number of peaks for initializing
     * @param num the number of scans
     * @param msLevel the MS-level of the scan
     * @param retentionTime the retention time
     * @param lowMz the lowest m/z value of the scan
     * @param highMz the highest m/z value of the scan
     * @param basePeakMz the m/z value of the base peak
     * @param basePeakIntensity the intensity of the base peak
     * @param totIonCurrent the total ion current of the scan
     * @param polarity the polarity according to definitions in CgDefines
     */
    public CgScan(int peaksCount, int num, int msLevel, float retentionTime, float lowMz, float highMz, float basePeakMz, float basePeakIntensity,
                    float totIonCurrent, int polarity){
      this (peaksCount);
      this.Num = num;
      this.MsLevel = msLevel;
      this.RetentionTime = retentionTime;
      this.LowMz = lowMz;
      this.HighMz = highMz;
      this.BasePeakMz = basePeakMz;
      this.BasePeakIntensity = basePeakIntensity;
      this.TotIonCurrent = totIonCurrent;
      this.polarity_ = polarity;
    }
    
    /**
     * constructor creating a clone of the current object
     * @param sc the other CgScan object to be cloned
     */
    public CgScan(CgScan sc){
      this(sc.PeaksCount, sc.Num, sc.MsLevel, sc.RetentionTime, sc.LowMz, sc.HighMz, sc.BasePeakMz, sc.BasePeakIntensity, sc.TotIonCurrent,sc.polarity_);
      this.Precision = sc.Precision;
      this.ByteOrder = sc.ByteOrder;
      this.PairOrder = sc.PairOrder;
      this.Scan = sc.Scan;
      this.SubScans = sc.SubScans;
      this.fullSubScans = sc.fullSubScans;
    }

	/**
     * Adds a subscan's mzXML scan number to the internal array
     *
	 * @param num
     *      a subscan's mzXML scan number
	 */
	public void AddSubscanNumber(int num)
	{
	    Integer i = new Integer(num);
	    SubScans.add(i);
	}

    public void AddSubscan(CgScan sc)
    {
        Integer i = new Integer(sc.Num);
        SubScans.add(i);
        fullSubScans.add(sc);
    }

	/**
     * Calculates the total intensity in a given m/z range
     *
	 * @param from
     *      lower m/z boundary
	 * @param to
     *      upper m/z boundary
	 * @return
     *      sum of intensities detected in the given range
	 */
	public float GetIntensity(float from, float to)
	{
	    float sum = 0;
    for (int i=0; i<PeaksCount; i++)
	    {
			if (Scan[i][0]>to) return sum;
			if (Scan[i][0]>from){
  sum += Scan[i][1];
}
	    }
	    return sum;
	}

  /**
   * fills an array of a fixed size with the intensity values inside a certain m/z range
   * this method is used by the RawToChromatogramTranslator to fill arrays of a certain batch size
   * every step in the array corresponds to the lowest resolution difference, e.g. 1mDa
   * @param array the array to be filled
   * @param from start m/z
   * @param to stop m/z
   * @param resolution the multiplication factor for int values, e.g. 1000 for 1mDa
   */
  public void fillIntensitiyArray(float[] array, float from, float to, int resolution){
    for (int i=0; i!=array.length;i++) array[i] = 0f;
    for (int j=0; j<PeaksCount; j+=100){
      if (j+100>=PeaksCount || Scan[j+100][0]>from){
        for (int k=j; k!=j+100; k+=10){
          if (k+10>=PeaksCount || Scan[k+10][0]>from){
            for (int i=k; i<PeaksCount; i++)
            {
              int intensityNr = Math.round(Calculator.roundFloat((Scan[i][0]-from)*(float)resolution, 0,BigDecimal.ROUND_UP));
              if (intensityNr>resolution || Scan[i][0]>to) return;
              if (intensityNr>0){
                intensityNr--;
                array[intensityNr] += Scan[i][1];
              }
              if (i==(PeaksCount-1))return;
            }
          }  
        }
      }
    }
  }

	/**
     * Determines whether a scan with the given scan number is
     * a subscan of this scan or the scan itself.
     *
	 * @param scanNumber
     *      The mzXML Scan Number to check
	 * @return
     *      true, if Scan number was found
	 */
	public boolean Contains(int scanNumber)
	{
	    int			i;

	    if (this.Num==scanNumber) return true;
	    for (i=0; i<SubScans.size(); i++)
	    {
	        if (SubScans.get(i).intValue()==scanNumber) return true;
	    }
	    return false;
	}

  public ArrayList<CgScan> getFullSubScans()
  {
    return fullSubScans;
  }

  public float[] getIntensities() {
	  float intensities[] = new float[PeaksCount];
	  for (int i=0; i<PeaksCount; i++)
		    intensities[i] = Scan[i][1];
	  return intensities;
  }

  public float[] getMZValues() {
	  float intensities[] = new float[PeaksCount];
	  for (int i=0; i<PeaksCount; i++)
		    intensities[i] = Scan[i][0];
	  return intensities;
  }

  /**
   * 
   * @return polarity according to the definition in CgDefines
   */
  public int getPolarity()
  {
    return polarity_;
  }

  /**
   * sets the polarity of a scan
   * @param polarity according to the definition in CgDefines
   */
  public void setPolarity(int polarity)
  {
    this.polarity_ = polarity;
  }

  public boolean isDummyScan()
  {
    return dummyScan_;
  }

  public void setDummyScan(boolean dummyScan)
  {
    this.dummyScan_ = dummyScan;
  }
  
  public void cleanSubscans(){
    SubScans = new ArrayList<Integer>();
    fullSubScans = new ArrayList<CgScan>();
  }

}
