package at.tugraz.genome.maspectras.chromaviewer;

import at.tugraz.genome.dbutilities.Base64;
import at.tugraz.genome.maspectras.quantification.*;
import at.tugraz.genome.maspectras.quantification.vos.*;
import java.awt.*;
import java.awt.event.*;
import java.nio.ByteBuffer;
import java.util.*;

public class CgChromaSetInfo
{
  private CgChromaViewer m_viewer;

  public Cg2dGraphicsManager m_2dPnl;

  private Cg3dGraphicsManager m_3dPnl;

  private CgChromatogramSet m_set;

  private Vector m_userProbes;

  private int m_charge; // Charge

  private int m_chromaCount; // Chromatogram count

  private int m_maxPoints; // maximum point count

  private float m_minTime, m_maxTime; // Retention Time boundaries

  private float m_minMz, m_maxMz; // Mz boundaries
  
  private float m_minRawMz, m_maxRawMz; // Mz boundaries

  private float m_maxIntensity; // Maximum Intensity (smoothed or raw)

  private float m_maxIntensitySmooth; // Maximum Intensity (smoothed)

  private float m_maxIntensityRaw; // Maximum Intensity (raw)

  private int m_curIndex; // Current Detail Chromatogram

  private boolean m_raw;

  private float m_minDispTime2d;

  private float m_maxDispTime2d;

  private float m_minDispTime3d;

  private float m_maxDispTime3d;

  private float m_3dGain; // Gain for 3d Display

  private float m_2dGain; // Gain for 2d Display

  private Image m_img;

  private Vector removedSystemProbes = new Vector();

  boolean differencesInStepAndSpan;

  public CgChromaSetInfo(CgChromaViewer viewer, Cg2dGraphicsManager p2d,
      Cg3dGraphicsManager p3d, int charge, CgChromatogramSet set)
  {
    m_viewer = viewer;
    m_2dPnl = p2d;
    m_3dPnl = p3d;
    m_charge = charge;
    m_set = set;
    m_curIndex = 0;
    m_raw = false;
    m_3dGain = 1;
    m_2dGain = 1;
    m_userProbes = new Vector(10);
    differencesInStepAndSpan = true;
    if (m_set == null)
      return;
    refreshProbes();
    calcBoundaries();
    zoomAll();
  }

  private void refreshProbes()
  {
    CgProbe cp, cq, cpRef;
    CgChromatogramVO cr;
    CgChromatogramPointVO pt;
    int i, j, k;

    // Refresh the "Duplicate" information

    if (m_set == null)
      return;
    if (m_set.getCgProbes().size() == 0)
      return;
    Vector newProbes = new Vector();
    for (i = 0; i < m_set.getCgProbes().size(); i++) {
      cp = (CgProbe) m_set.getCgProbes().get(i);
      for (k = 0; k < m_set.getChromatograms().size(); k++) {
        cr = (CgChromatogramVO) m_set.getChromatograms().get(k);
        if (cp.Mz < (cr.getMz() + this.getThreshold())
            && cp.Mz > cr.getMz() - this.getThreshold()) {
          CgAreaDeterminer det = new CgAreaDeterminer(cr, cp.Peak, m_set.getAutomaticQuantMethod().intValue());
          cq = det.getCgProbe();
          cq.Mz = cp.Mz;
          cq.UpperMzBand = cp.UpperMzBand;
          cq.LowerMzBand = cp.LowerMzBand;
          cq.Charge = cp.Charge;
          cp = cq;
          boolean duplicate = false;
          for (j = 0; j < newProbes.size(); j++) {
            CgProbe px = (CgProbe) newProbes.get(j);
            //if (cp.EqualValleys(cpRef) == true)
            if (px.EqualValleys(cp) == true)
              duplicate = true;
            else{
              if (px.isCoveredByThisProbe(cp))
                duplicate = true;
              if (cp.isCoveredByThisProbe(px))
                duplicate = true;   
            }
            if (duplicate)
              break;
          }
          if (!duplicate) {
            newProbes.add(cp);
          }
        }
      }
    }
    m_set.setCgProbes(newProbes);
/*    for (i = 0; i < m_set.getCgProbes().size(); i++) {
      cp = (CgProbe) m_set.getCgProbes().get(i);

      // if (cp.AreaStatus==CgAreaStatus.Duplicate) cp.AreaStatus =
      // CgAreaStatus.OK;
    }*/
    for (i = 0; i < m_set.getCgProbes().size(); i++) {
      cpRef = (CgProbe) m_set.getCgProbes().get(i);

      if (cpRef.AreaStatus != CgAreaStatus.OK)
        continue;

      for (j = i + 1; j < m_set.getCgProbes().size(); j++) {
        cp = (CgProbe) m_set.getCgProbes().get(j);
        if (cp.EqualValleys(cpRef) == true)
          cpRef.AreaStatus = CgAreaStatus.Duplicate;
        else{
          if (cp.isCoveredByThisProbe(cpRef))
            cp.AreaStatus = CgAreaStatus.Duplicate;
          if (cpRef.isCoveredByThisProbe(cp))
            cpRef.AreaStatus = CgAreaStatus.Duplicate;         
        }
/*        if (cp.LowerValley == cpRef.LowerValley
            && cp.UpperValley == cpRef.UpperValley) {
          cp.AreaStatus = CgAreaStatus.Duplicate;
        }*/
      }
    }

    // Set the CgProbe's Scan to the corresponding index in the Chromatogram:

    for (i = 0; i < m_set.getCgProbes().size(); i++) {
      cp = (CgProbe) m_set.getCgProbes().get(i);
      if (cp.AreaStatus == CgAreaStatus.OK)
        continue;
      cr = null;
      for (j = 0; j < m_set.getChromatograms().size(); j++) {
        cr = (CgChromatogramVO) m_set.getChromatograms().get(j);
        if (cr.getMz() >= (cp.Mz - this.getThreshold())
            && cr.getMz() <= (cp.Mz + this.getThreshold()))
          break;
        cr = null;
      }
      if (cr == null)
        continue;
      for (j = 0; j < cr.getChromatogramPointVOs().size(); j++) {
        pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(j);
        if (pt.getScan() >= cp.Peak) {
          cp.Scan = j;
          break;
        }
      }
    }
    Vector myOldProbes = new Vector();
    for (i = 0; i < m_set.getOldProbes().size(); i++) {
      cp = (CgProbe) m_set.getOldProbes().get(i);
      // if (cp.AreaStatus==CgAreaStatus.OK) continue;
      cr = null;
      for (j = 0; j < m_set.getChromatograms().size(); j++) {
        cr = (CgChromatogramVO) m_set.getChromatograms().get(j);
        if (cr.getMz() >= (cp.Mz - this.getThreshold())
            && cr.getMz() <= (cp.Mz + this.getThreshold()))
          break;
        cr = null;
      }
      if (cr == null)
        continue;
      for (j = 0; j < cr.getChromatogramPointVOs().size(); j++) {
        pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(j);
        if (pt.getScan() >= cp.Peak) {
          cp.Scan = j;
          break;
        }
      }
      myOldProbes.add(cp);
    }
    m_set.setOldProbes(myOldProbes);

    // if the stepSize and the spans are the same we can do additional things
    // put removed areas to the removedList and add an Id to the calculated
    // Probes:

    if ((new Double(m_viewer.m_stepMzText.getText()).doubleValue()) >= (m_set
        .getStep().doubleValue() - this.getThreshold())
        && (new Double(m_viewer.m_stepMzText.getText()).doubleValue()) <= (m_set
            .getStep().doubleValue() + this.getThreshold())
        && (new Double(m_viewer.m_lowerSpanText.getText()).doubleValue()) >= (m_set
            .getLowerSpan().doubleValue() - this.getThreshold())
        && (new Double(m_viewer.m_lowerSpanText.getText()).doubleValue()) <= (m_set
            .getLowerSpan() + this.getThreshold())
        && (new Double(m_viewer.m_upperSpanText.getText()).doubleValue()) >= (m_set
            .getUpperSpan().doubleValue() - this.getThreshold())
        && (new Double(m_viewer.m_upperSpanText.getText()).doubleValue()) <= (m_set
            .getUpperSpan() + this.getThreshold())) {
      differencesInStepAndSpan = false;

      Vector myProbes = new Vector();

      for (i = 0; i != m_set.getCgProbes().size(); i++) {
        CgProbe calcProbe = (CgProbe) m_set.getCgProbes().get(i);
        boolean exists = false;
        for (j = 0; j != m_set.getOldProbes().size(); j++) {
          cp = (CgProbe) m_set.getOldProbes().get(j);
          if (this.sameCgProbe(calcProbe, cp)) {
            exists = true;
            calcProbe.setPeakId(cp.getPeakId());
          }
        }
        if (!exists){
          removedSystemProbes.add(calcProbe);
        }else{
          myProbes.add(calcProbe);
        }  
      }

      for (j = 0; j != m_set.getOldProbes().size(); j++) {
        cp = (CgProbe) m_set.getOldProbes().get(j);
        boolean exists = false;
        CgProbe calcProbe = null;
        for (i = 0; i != m_set.getCgProbes().size(); i++) {
          calcProbe = (CgProbe) m_set.getCgProbes().get(i);
          if (this.sameCgProbe(calcProbe, cp)) {
            exists = true;
          }
        }
        if (!exists)
          myProbes.add(cp);
      }
      m_set.setCgProbes(myProbes);
    }
  }

  public float get2dGain()
  {
    return m_2dGain;
  }

  public void set2dGain(float value)
  {
    m_2dGain = value;
  }

  public float get3dGain()
  {
    return m_3dGain;
  }

  public void set3dGain(float value)
  {
    m_3dGain = value;
  }

  public void set2dShift(int direction)
  {
    float x, diff, delta;
    diff = m_maxDispTime2d - m_minDispTime2d;
    delta = diff / 40;
    if (direction > 0) {
      x = m_minDispTime2d - delta;
      if (x < 0)
        x = 0;
      m_minDispTime2d = x;
      m_maxDispTime2d = x + diff;
    } else {
      x = m_maxDispTime2d + delta;
      if (x > m_maxTime)
        x = m_maxTime;
      m_maxDispTime2d = x;
      m_minDispTime2d = x - diff;
    }
  }

  public boolean setMinDispTime2d(float newVal)
  {
    if (newVal < 0)
      return false;
    if (newVal < m_minTime)
      newVal = m_minTime;
    m_minDispTime2d = newVal;
    if (m_minDispTime2d < m_maxTime / 100)
      m_minDispTime2d = 0;
    return true;
  }

  public float getMinDispTime()
  {
    return m_minDispTime2d;
  }

  public boolean setMaxDispTime2d(float newVal)
  {
    if (newVal < m_minDispTime2d)
      return false;
    if (newVal > m_maxTime)
      newVal = m_maxTime;
    m_maxDispTime2d = newVal;
    return true;
  }

  public float getMaxDispTime()
  {
    return m_maxDispTime2d;
  }

  public void zoomAll()
  {
    m_minDispTime2d = m_minTime;
    m_maxDispTime2d = m_maxTime;
    if (m_minDispTime2d < m_maxTime / 100)
      m_minDispTime2d = 0;
  }

  public int getChromaCount()
  {
    return m_chromaCount;
  }

  public Image getImage()
  {
    return m_img;
  }

  public void setPreviousChromatogram()
  {
    m_curIndex--;
    if (m_curIndex <= 0)
      m_curIndex = 0;
  }

  public void setNextChromatogram()
  {
    m_curIndex++;
    if (m_curIndex >= m_chromaCount)
      m_curIndex = m_chromaCount - 1;
  }

  private boolean in2dPanel = false;

  public void mouse2dAction(Cg2dGraphicsManager m, MouseEvent e)
  {
    float val = 123;
    switch (e.getID()) {
      case MouseEvent.MOUSE_ENTERED:
        m.setCrosshairCursor();
        in2dPanel = true;
        break;
      case MouseEvent.MOUSE_MOVED:
        m.setIntensity(getIntensity(e.getY()));
        m.setRetentionTime(getRetentionTime(e.getX()));
        break;
      case MouseEvent.MOUSE_EXITED:
        m.setDefaultCursor();
        m.clearIntensity();
        m.clearRetentionTime();
        in2dPanel = false;
        break;
    }
  }

  public void popup2dAction(ActionEvent e)
  {
    CgProbe cx;
    if (e.getActionCommand() == "Determine Area") {
      cx = getArea(m_2dPnl.lastPopupX, 0);
      removeOverlappingProbes(cx);
      if (restoreSystemArea(cx) == false)
        determineArea(m_2dPnl.lastPopupX, 0);
      return;
    }
    if (e.getActionCommand() == "Determine Area (Col)") {
      cx = getArea(m_2dPnl.lastPopupX, 1);
      removeOverlappingProbes(cx);
      if (restoreSystemArea(cx) == false)
        determineArea(m_2dPnl.lastPopupX, 1);
      return;
    } else if (e.getActionCommand() == "Delete Area") {
      if (this.anySystemCgProbeThere(m_2dPnl.lastPopupX))
        removeSystemArea(m_2dPnl.lastPopupX);
      else
        deleteArea(m_2dPnl.lastPopupX);
    }
  }

  public float getRetentionTime(int x)
  {
    if (x < leftMargin_2d())
      return -1;
    int w0 = m_2dPnl.getWidth() - leftMargin_2d() - rightMargin_2d();
    if (x > leftMargin_2d() + w0)
      return -1;
    return m_minDispTime2d
        + ((x - leftMargin_2d()) * (m_maxDispTime2d - m_minDispTime2d))
        / (float) w0;
  }

  public float getIntensity(int y)
  {
    if (y < topMargin_2d())
      return -1;
    int h0 = m_2dPnl.getHeight() - topMargin_2d() - bottomMargin_2d();
    if (y > (topMargin_2d() + h0))
      return -1;
    return (h0 + topMargin_2d() - y) * m_maxIntensity / m_2dGain / h0;
  }

  public boolean anyUserCgProbeThere(int x)
  {
    CgProbe cp;

    CgChromatogramVO cr = (CgChromatogramVO) m_set.getChromatograms().get(
        m_curIndex);
    float t = getRetentionTime(x);
    for (int i = 0; i < m_userProbes.size(); i++) {
      cp = (CgProbe) m_userProbes.get(i);
      if (cp.LowerValley > t || cp.UpperValley < t)
        continue;
      if (cp.Mz > (cr.getMz() + this.getThreshold())
          || cp.Mz < (cr.getMz() - this.getThreshold()))
        continue;
      return true;
    }
    return false;
  }

  public boolean anySystemCgProbeThere(int x)
  {
    CgProbe cp;

    CgChromatogramVO cr = (CgChromatogramVO) m_set.getChromatograms().get(
        m_curIndex);
    float t = getRetentionTime(x);
    for (int i = 0; i < m_set.getCgProbes().size(); i++) {
      cp = (CgProbe) ((m_set.getCgProbes()).get(i));
      if (cp.LowerValley > t || cp.UpperValley < t)
        continue;
      if (cp.Mz > (cr.getMz() + this.getThreshold())
          || cp.Mz < (cr.getMz() - this.getThreshold()))
        continue;
      boolean removed = false;
      for (int j = 0; j != removedSystemProbes.size(); j++) {
        if (this.sameCgProbe((CgProbe) removedSystemProbes.get(j), cp))
          removed = true;
      }
      return (!removed);
    }
    return false;
  }

  // This method looks through the System Probes and the User Probes and removes
  // any probe,
  // that overlaps with the existing probe. Overlapping System Probes are set
  // onto the ex-
  // clusion list, overlapping user probes are deleted.

  private void removeOverlappingProbes(CgProbe cx)
  {
    int i, j;
    CgProbe cp, cq;
    boolean loop;
    boolean alreadyExcluded;

    // Checking of System Probes. If any system probe there and it isn't on the
    // exclusion list already, we put it there.
    for (i = 0; i < m_set.getCgProbes().size(); i++) {
      cp = (CgProbe) m_set.getCgProbes().get(i);
      if ((cp.Mz > (cx.Mz - this.getThreshold()))
          && (cp.Mz < (cx.Mz + this.getThreshold()))) {
        if ((cp.UpperValley < cx.LowerValley)
            && (cp.LowerValley > cx.UpperValley)) {

          // They seem the same. Now let's have a look wheter we are already on
          // the
          // exclusion list:
          alreadyExcluded = false;
          for (j = 0; j < removedSystemProbes.size(); j++) {
            cq = (CgProbe) removedSystemProbes.get(i);
            if (sameCgProbe(cp, cq) == true) {
              alreadyExcluded = true;
              break;
            }
          }
          if (alreadyExcluded == false)
            removedSystemProbes.add(cp);
        }
      }
    }

    // Cheching of User Probes. If any user probe is out there, we just delete
    // it.

    loop = true;
    while (loop) {
      loop = false;
      for (i = 0; i < m_userProbes.size(); i++) {
        cp = (CgProbe) m_userProbes.get(i);
        if ((cp.Mz > (cx.Mz - this.getThreshold()))
            && (cp.Mz < (cx.Mz + this.getThreshold()))) {
          if (cp.UpperValley < cx.LowerValley)
            continue;
          if (cp.LowerValley < cx.UpperValley)
            continue;
          m_userProbes.remove(i);
          loop = true;
        }
      }
    }
  }

  private boolean restoreSystemArea(CgProbe cx)
  {
    for (int i = 0; i < this.removedSystemProbes.size(); i++) {
      CgProbe cp = (CgProbe) ((this.removedSystemProbes).get(i));
      if (this.sameCgProbe(cp, cx) == true) {
        // We have to check wheter this is really one that we can find in the
        // cgProbes...

        for (int j = 0; j < m_set.getCgProbes().size(); j++) {
          CgProbe cq = (CgProbe) m_set.getCgProbes().get(j);
          if (this.sameCgProbe(cp, cq) == true) {
            this.removedSystemProbes.remove(i);
            m_2dPnl.preparePaint();
            Draw2DDiagram(m_2dPnl, m_raw);
            this.refresh3dGraphics();
            return true;
          }
        }
      }
    }
    return false;
  }

  public boolean anyRemovedCgProbeThere(int x, CgProbe cx)
  {
    CgProbe cp;

    float t = getRetentionTime(x);
    for (int i = 0; i < this.removedSystemProbes.size(); i++) {
      cp = (CgProbe) ((this.removedSystemProbes).get(i));
      if (this.sameCgProbe(cp, cx) == true)
        return true;
    }
    return false;
  }

  public int checkPopupClick(int x, int y)
  {
    float ix, z, n;

    CgChromatogramPointVO pt1, pt2;

    if (in2dDiagram(x, y) == false)
      return 0;

    float t = getRetentionTime(x);
    float intensity = getIntensity(y);

    CgChromatogramVO cr = (CgChromatogramVO) m_set.getChromatograms().get(
        m_curIndex);
    for (int i = 1; i < cr.getChromatogramPointVOs().size(); i++) {
      pt1 = (CgChromatogramPointVO) (cr.getChromatogramPointVOs().get(i - 1));
      pt2 = (CgChromatogramPointVO) (cr.getChromatogramPointVOs().get(i));
      if (pt2.getScan() < t)
        continue;
      n = pt2.getScan() - pt1.getScan();
      if (m_raw == false) {
        z = pt2.getSmoothedIntensity() - pt1.getSmoothedIntensity();
        ix = pt1.getSmoothedIntensity() + z / n * (t - pt1.getScan());
      } else {
        z = pt2.getIntensity() - pt1.getIntensity();
        ix = pt1.getIntensity() + z / n * (t - pt1.getScan());
      }
      if (intensity < ix)
        return 1;
      return 0;
    }
    return 0;
  }

  private void deleteArea(int x)
  {
    CgProbe cp;
    CgChromatogramVO cr = (CgChromatogramVO) (m_set.getChromatograms()
        .get(m_curIndex));
    float t = getRetentionTime(x);
    for (int i = 0; i < this.m_userProbes.size(); i++) {
      cp = (CgProbe) ((this.m_userProbes).get(i));
      if (cp.LowerValley > t || cp.UpperValley < t)
        continue;
      if (cp.Mz > (cr.getMz() + this.getThreshold())
          || cp.Mz < (cr.getMz() - this.getThreshold()))
        continue;
      m_userProbes.remove(cp);
      break;
    }
    m_2dPnl.preparePaint();
    Draw2DDiagram(m_2dPnl, m_raw);
    this.refresh3dGraphics();

  }
  
  private void refresh3dGraphics(){
    if (m_viewer.m_chkQuasi3D.getState()){
      Draw3DDiagram(m_3dPnl, m_raw);
      m_3dPnl.refresh();
    }else{
      m_viewer.viewer.setPaintableProbes(this.getProbesToPaint());
      m_viewer.viewer.setTheShowSelectedWasOn(false);
      m_viewer.viewer.repaintColors();
    }
  }

  private void removeSystemArea(int x)
  {
    CgProbe cp;
    CgChromatogramVO cr = (CgChromatogramVO) (m_set.getChromatograms()
        .get(m_curIndex));
    float t = getRetentionTime(x);
    for (int i = 0; i < this.m_set.getCgProbes().size(); i++) {
      cp = (CgProbe) ((this.m_set.getCgProbes()).get(i));
      if (cp.LowerValley > t || cp.UpperValley < t)
        continue;
      if (cp.Mz > (cr.getMz() + this.getThreshold())
          || cp.Mz < (cr.getMz() - this.getThreshold()))
        continue;
      this.removedSystemProbes.add(cp);
      break;
    }
    m_2dPnl.preparePaint();
    Draw2DDiagram(m_2dPnl, m_raw);
    this.refresh3dGraphics();
  }

  private void addSystemArea(int x)
  {
    CgProbe cp;
    CgChromatogramVO cr = (CgChromatogramVO) (m_set.getChromatograms()
        .get(m_curIndex));
    float t = getRetentionTime(x);
    Vector newRemovedProbes = new Vector();
    for (int i = 0; i < this.removedSystemProbes.size(); i++) {
      cp = (CgProbe) (this.removedSystemProbes.get(i));
      if (cp.LowerValley > t || cp.UpperValley < t)
        continue;
      if (cp.Mz > (cr.getMz() + this.getThreshold())
          || cp.Mz < (cr.getMz() - this.getThreshold()))
        continue;
      this.removedSystemProbes.remove(cp);
      break;
    }
    m_2dPnl.preparePaint();
    Draw2DDiagram(m_2dPnl, m_raw);
    this.refresh3dGraphics();

  }

  private CgProbe getArea(int x, int method)
  {
    float t = getRetentionTime(x);
    CgChromatogramVO cr = (CgChromatogramVO) (m_set.getChromatograms()
        .get(m_curIndex));
    CgAreaDeterminer det = new CgAreaDeterminer(cr, t, method);
    CgProbe cx = det.getCgProbe();
    cx.LowerMzBand = cr.getLowerMzBand();
    cx.Mz = cr.getMz();
    cx.UpperMzBand = cr.getUpperMzBand();
    cx.Charge = m_charge;
    return cx;
  }

  private void determineArea(int x, int method)
  {
    float t = getRetentionTime(x);
    CgChromatogramVO cr = (CgChromatogramVO) (m_set.getChromatograms()
        .get(m_curIndex));
    CgAreaDeterminer det = new CgAreaDeterminer(cr, t, method);
    CgProbe cx = det.getCgProbe();
    cx.LowerMzBand = cr.getLowerMzBand();
    cx.Mz = cr.getMz();
    cx.UpperMzBand = cr.getUpperMzBand();
    cx.Charge = m_charge;
    if (cx.AreaStatus != CgAreaStatus.OK)
      return;
    m_userProbes.add(cx);
    Draw2DDiagram(m_2dPnl, m_raw);
    this.refresh3dGraphics();
  }

  public void Draw2DDiagram(Cg2dGraphicsManager m, boolean raw)
  {
    CgChromatogramVO cr;
    CgChromatogramPointVO pt;

    int x0, y0, w0, h0;
    int xx = 0, yy = 0;
    int x = 0, y = 0;
    int dx, dy;
    int i, j, len;
    boolean firstPoint;
    String s;

    Polygon p;

    if (m_set == null)
      return;

    Graphics gx = m.getGraphics();
    m_raw = raw;
    gx.setColor(new Color(240, 240, 240));
    gx.drawRect(0, 0, m.getWidth() - 1, m.getHeight() - 1);
    if (m_set == null)
      return;

    if (m_curIndex < 0)
      m_curIndex = 0;
    if (m_curIndex >= m_chromaCount)
      m_curIndex = m_chromaCount - 1;
    if (m_chromaCount == 0)
      return;

    w0 = m.getWidth() - leftMargin_2d() - rightMargin_2d();
    h0 = m.getHeight() - topMargin_2d() - bottomMargin_2d();

    x0 = leftMargin_2d();
    y0 = m.getHeight() - bottomMargin_2d();

    // **** Draw the Areas ****

    cr = (CgChromatogramVO) m_set.getChromatograms().get(m_curIndex);
    len = cr.getChromatogramPointVOs().size();
    Vector paintableProbes = new Vector();

    for (i = 0; i != m_set.getCgProbes().size(); i++) {
      CgProbe aProbe = (CgProbe) m_set.getCgProbes().get(i);
      boolean removed = false;
      for (j = 0; j != this.removedSystemProbes.size(); j++) {
        if (this.sameCgProbe((CgProbe) this.removedSystemProbes.get(j), aProbe)) {
          removed = true;
        }
      }
      if (!removed)
        paintableProbes.add(aProbe);
    }

    paint2dAreas(cr, paintableProbes, Color.RED);
    paint2dAreas(cr, this.m_userProbes, Color.GREEN);
    paint2dOldAreas(cr, m_set.getOldProbes());

    // **** Draw the diagram ****

    gx.setColor(Color.BLACK);
    firstPoint = true;
    for (i = 0; i < len; i++) {
      pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(i);

      if (pt.getScan() < m_minDispTime2d)
        continue;
      if (pt.getScan() > m_maxDispTime2d)
        continue;

      x = x0
          + (int) (w0 * (pt.getScan() - m_minDispTime2d) / (m_maxDispTime2d - m_minDispTime2d));

      if (raw == true)
        y = y0 - (int) (h0 * pt.getIntensity() * m_2dGain / m_maxIntensity);
      else
        y = y0
            - (int) (h0 * pt.getSmoothedIntensity() * m_2dGain / m_maxIntensity);
      if (firstPoint == true) {
        xx = x;
        yy = y;
        firstPoint = false;
      }
      gx.drawLine(xx, yy, x, y);
      xx = x;
      yy = y;
    }

    // **** Draw the x coordinates ****

    float t = 0;
    float dt;

    gx.setFont(new Font("SansSerif", Font.PLAIN, 10));
    FontMetrics fm = gx.getFontMetrics();
    int sw;

    x0 = leftMargin_2d();
    y0 = m.getHeight() - bottomMargin_2d();

    gx.drawLine(x0, y0, x0 + w0 + 3, y0);
    gx.drawLine(x0 + w0 + 3, y0 - 2, x0 + w0 + 3, y0 + 2);
    gx.drawLine(x0 + w0 + 3, y0 - 2, x0 + w0 + 9, y0);
    gx.drawLine(x0 + w0 + 3, y0 + 2, x0 + w0 + 9, y0);

    gx.drawLine(x0, y0, x0, y0 - h0 + 5);
    gx.drawLine(x0 - 2, y0 - h0 + 5, x0 + 2, y0 - h0 + 5);
    gx.drawLine(x0 - 2, y0 - h0 + 5, x0, y0 - h0 - 1);
    gx.drawLine(x0 + 2, y0 - h0 + 5, x0, y0 - h0 - 1);

    dt = 100;
    t = (m_maxDispTime2d - m_minDispTime2d) / 10;
    if (t < 10)
      dt = 10;
    else if (t < 20)
      dt = 20;
    else if (t < 50)
      dt = 50;
    else if (t < 100)
      dt = 100;
    else if (t < 200)
      dt = 200;
    else if (t < 500)
      dt = 500;
    else if (t < 1000)
      dt = 1000;
    else if (t < 2000)
      dt = 2000;
    else if (t < 5000)
      dt = 5000;
    else if (t < 10000)
      dt = 10000;
    else if (t < 20000)
      dt = 20000;
    else if (t < 50000)
      dt = 50000;
    else
      dt = 100000;

    j = fm.stringWidth("Time / s");

    i = (int) (m_minDispTime2d / dt);
    if (i > 0)
      t = (i - 1) * dt;
    else
      t = 0;
    while (t < m_maxDispTime2d) {
      if (t >= m_minDispTime2d) {
        x = x0
            + (int) ((w0 * (t - m_minDispTime2d)) / (m_maxDispTime2d - m_minDispTime2d));
        y = y0;
        gx.drawLine(x, y, x, y + 3);
        s = Integer.toString((int) t);
        sw = fm.stringWidth(s);

        if ((x + sw / 2 + 10) < (x0 + w0 - j))
          gx.drawString(s, x - sw / 2, y + 12);
      }
      t += dt;
    }
    gx.drawString("Time / s", x0 + w0 - j + 9, y + 12);

    // **** Draw the y coordinates ****

    float in = 0;
    float din = 0;

    in = m_maxIntensity / m_2dGain / 5;

    if (in < 10)
      din = 10;
    else if (in < 25)
      din = 25;
    else if (in < 50)
      din = 50;
    else if (in < 100)
      din = 100;
    else if (in < 250)
      din = 250;
    else if (in < 500)
      din = 500;
    else if (in < 1000)
      din = 1000;
    else if (in < 2500)
      din = 2500;
    else if (in < 5000)
      din = 5000;
    else if (in < 10000)
      din = 10000;
    else if (in < 25000)
      din = 25000;
    else if (in < 50000)
      din = 50000;
    else if (in < 100000)
      din = 100000;
    else if (in < 250000)
      din = 250000;
    else if (in < 500000)
      din = 500000;
    else if (in < 1000000)
      din = 1000000;
    else if (in < 2500000)
      din = 2500000;
    else if (in < 5000000)
      din = 5000000;
    else if (in < 10000000)
      din = 10000000;
    else if (in < 25000000)
      din = 25000000;
    else if (in < 50000000)
      din = 50000000;
    else if (in < 100000000)
      din = 100000000;
    else if (in < 250000000)
      din = 250000000;
    else if (in < 500000000)
      din = 500000000;
    else
      din = 1000000000;

    j = fm.stringWidth("Arb.Units / AU");
    in = 0;
    y = y0 - (int) (h0 * in / m_maxIntensity * m_2dGain);
    // while(in<m_maxIntensity)
    while (y > (y0 - h0)) {
      x = x0;
      y = y0 - (int) (h0 * in / m_maxIntensity * m_2dGain);
      if (y < (y0 - h0 + 10))
        break;
      if (in > 2147483000)
        break;

      gx.drawLine(x, y, x - 3, y);
      s = Integer.toString((int) in);
      sw = fm.stringWidth(s);
      if (in == 0)
        gx.drawString(s, x - sw - 4, y);
      else
        gx.drawString(s, x - sw - 4, y + 4);
      in += din;
    }
    gx.drawString("Arb.Units / AU", x - j - 4, y0 - h0 + 6);
    m.setMz(cr.getMz());
  }

  private void paint2dAreas(CgChromatogramVO cr, Vector areas, Color col)
  {
    int i, j, len;
    int x = 0, y = 0;
    boolean firstPoint;
    Polygon p;
    String s;
    CgChromatogramPointVO pt;

    int x0 = leftMargin_2d();
    int w0 = diagramWidth_2d();
    int y0 = topMargin_2d() + diagramHeight_2d();
    int h0 = diagramHeight_2d();

    Graphics gx = m_2dPnl.getGraphics();

    len = cr.getChromatogramPointVOs().size();
    for (i = 0; i < areas.size(); i++) {
      CgProbe cp = (CgProbe) areas.get(i);
      if (cp == null)
        continue;
      if (cp.AreaStatus != CgAreaStatus.OK)
        continue;
      if (cp.Mz < (cr.getMz() + this.getThreshold())
          && cp.Mz > (cr.getMz() - this.getThreshold())) {
        // Paint the Area of the Peak accordingly:

        p = new Polygon();
        firstPoint = true;
        for (j = 0; j < len; j++) {
          pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(j);

          if (pt.getScan() < m_minDispTime2d)
            continue;
          if (pt.getScan() < cp.LowerValley)
            continue;
          if (pt.getScan() > m_maxDispTime2d)
            continue;
          if (pt.getScan() > cp.UpperValley)
            continue;

          x = x0
              + (int) (w0 * (pt.getScan() - m_minDispTime2d) / (m_maxDispTime2d - m_minDispTime2d));

          if (m_raw == true)
            y = y0 - (int) (h0 * pt.getIntensity() * m_2dGain / m_maxIntensity);
          else
            y = y0
                - (int) (h0 * pt.getSmoothedIntensity() * m_2dGain / m_maxIntensity);

          if (firstPoint == true) {
            p.addPoint(x, y0);
            firstPoint = false;
          }
          p.addPoint(x, y);
        }
        if (p.npoints > 0) {
          p.addPoint(x, y0);
          gx.setColor(col);
          gx.fillPolygon(p);

          // draw the Area:

          x = x0
              + (int) (w0 * (cp.Peak - m_minDispTime2d) / (m_maxDispTime2d - m_minDispTime2d));

          pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs()
              .get(cp.Scan);

          if (m_raw == true)
            y = y = y0
                - (int) (h0 * pt.getIntensity() * m_2dGain / m_maxIntensity)
                - 5;
          else
            y = y = y0
                - (int) (h0 * pt.getSmoothedIntensity() * m_2dGain / m_maxIntensity)
                - 5;

          if (y < 30) {
            y = 30;
            x += 10;
          }

          s = String.format("A=%4.3e", cp.Area);
          s = s.replace("+", "")
              + String.format(" @ %.0f<t<%.0f", cp.LowerValley, cp.UpperValley);
          s += String.format(" / -%.2fd+%.2fd", cp.LowerMzBand, cp.UpperMzBand);
          s += String.format(" / Bckgnd/A=%.1f%%", 100 * cp.Background
              / cp.Area);
          s = s.replace("e0", "e").replace(",", ".");
          gx.setColor(Color.BLUE);
          gx.drawString(s, x, y);
        }
      }
    }
  }

  private void paint2dOldAreas(CgChromatogramVO cr, Vector areas)
  {
    int i, j, len;
    int x = 0, y = 0;
    boolean firstPoint;
    Polygon p;
    String s;
    CgChromatogramPointVO pt;

    int x0 = leftMargin_2d();
    int w0 = diagramWidth_2d();
    int y0 = topMargin_2d() + diagramHeight_2d();
    int h0 = diagramHeight_2d();

    Graphics gx = m_2dPnl.getGraphics();

    len = cr.getChromatogramPointVOs().size();
    for (i = 0; i < areas.size(); i++) {
      CgProbe cp = (CgProbe) areas.get(i);
      if (cp == null)
        continue;
      if (cp.AreaStatus != CgAreaStatus.OK)
        continue;
      boolean ok = false;
      if ((cp.Mz < (cr.getMz() + this.getThreshold()) && cp.Mz > (cr.getMz() - this
          .getThreshold())/* &&(!differencesInStepAndSpan))||differencesInStepAndSpan */)) {
        x = x0
            + (int) (w0 * (cp.Peak - m_minDispTime2d) / (m_maxDispTime2d - m_minDispTime2d));

        pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(cp.Scan);

        if (m_raw == true)
          y = y = y0
              - (int) (h0 * pt.getIntensity() * m_2dGain / m_maxIntensity) - 5;
        else
          y = y = y0
              - (int) (h0 * pt.getSmoothedIntensity() * m_2dGain / m_maxIntensity)
              - 5;

        if (y < 30) {
          y = 30;
          x += 10;
        }

        s = String.format("A=%4.3e", cp.Area);
        s = s.replace("+", "")
            + String.format(" @ %.0f<t<%.0f", cp.LowerValley, cp.UpperValley);
        s += String.format(" / -%.2fd+%.2fd", cp.LowerMzBand, cp.UpperMzBand);
        s += String.format(" / Bckgnd/A=%.1f%%", 100 * cp.Background / cp.Area);
        s = s.replace("e0", "e").replace(",", ".");
        gx.setColor(Color.BLACK);
        gx.drawString(s, x, y - 14);
      }
    }
  }

  public Vector getAddableProbes()
  {
    Vector addableProbes = new Vector();
    if (m_set != null) {
      if (m_set.getCgProbes() == null)
        return addableProbes;
      for (int i = 0; i != m_set.getCgProbes().size(); i++) {
        CgProbe cp = (CgProbe) m_set.getCgProbes().get(i);
        boolean removed = false;
        if (this.removedSystemProbes != null) {
          for (int j = 0; j != this.removedSystemProbes.size(); j++) {
            removed = this.sameCgProbe(cp, (CgProbe) this.removedSystemProbes
                .get(j));
          }
        }
        if (!removed)
          addableProbes.add(cp);
      }
      if (this.m_userProbes != null) {
        for (int i = 0; i != m_userProbes.size(); i++) {
          addableProbes.add((CgProbe) m_userProbes.get(i));
        }
      }
    }
    return addableProbes;
  }

  public void updateSystemProbes(Vector probes)
  {
    if (m_set != null) {
      m_set.setCgProbes(probes);
      m_set.setOldProbes(probes);
      this.m_userProbes = new Vector();
      this.removedSystemProbes = new Vector();
      // m_2dPnl.preparePaint();
      // m_3dPnl.preparePaint();
      // Draw2DDiagram(m_2dPnl, m_raw);
      // Draw3DDiagram(m_3dPnl, m_raw);
      // m_3dPnl.refresh();
    }
  }

  public int getSpanNumber(CgProbe cp, int lowerSpan)
  {
    int spanNumber = -1000;
    for (int i = 0; i < m_chromaCount; i++) {
      CgChromatogramVO cr = (CgChromatogramVO) m_set.getChromatograms().get(i);
      if (cp.Mz < (cr.getMz() + this.getThreshold())
          && cp.Mz > (cr.getMz() - this.getThreshold())) {
        return (i - lowerSpan);
      }
    }
    return spanNumber;
  }

  public Vector getDeletableProbes()
  {
    if (this.removedSystemProbes == null)
      return new Vector();
    return this.removedSystemProbes;
  }

  public void Draw3DDiagram(Cg3dGraphicsManager m, boolean raw)
  {
    int DELTA_X = 4;
    int DELTA_Y = 6;
//    int DELTA_X = 1;
//    int DELTA_Y = 1;
    
    int x0, y0, w0, h0;
    int xx, yy;
    int x, y;
    int dx, dy;
    int i, j, len;
    int ii, jj;
    Polygon p, pp;

    float maxTime;

    String s;
    int sw;

    CgChromatogramVO cr = null;
    CgChromatogramPointVO pt;

    if (m_set == null)
      return;

    m_raw = raw;
    Vector paintableProbes = this.getPaintableSystemProbes(); 

    Graphics gx = m.getBufferGraphics();
    m_3dPnl.resetArea();
    gx.setColor(Color.BLACK);
    gx.setFont(new Font("SansSerif", Font.PLAIN, 10));
    FontMetrics fm = gx.getFontMetrics();

    try {
      if (m_curIndex < 0)
        m_curIndex = 0;
      if (m_curIndex >= m_chromaCount)
        m_curIndex = m_chromaCount - 1;

      w0 = m.getWidth() - leftMargin_3d() - DELTA_X * (m_chromaCount - 1)
          - rightMargin_3d();
      h0 = m.getHeight() - topMargin_3d() - DELTA_Y * (m_chromaCount - 1)
          - bottomMargin_3d();

      x0 = leftMargin_3d() + DELTA_X * (m_chromaCount - 1);
      y0 = m.getHeight() - DELTA_Y * (m_chromaCount - 1) - bottomMargin_3d();
      xx = x0;
      yy = y0;

      // **** Draw the Diagram ****

      x = 0;
      y = 0;
      gx.setColor(Color.BLACK);
//      if (m_set.getRawData()==null){
      for (i = 0; i < m_chromaCount; i++) {
        cr = (CgChromatogramVO) m_set.getChromatograms().get(i);
        len = cr.getChromatogramPointVOs().size();
        pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(len - 1);
        maxTime = pt.getScan();
        p = new Polygon();
        p.addPoint(x0, y0);
        for (j = 0; j < len; j++) {
          pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(j);
          x = x0 + (int) (w0 * pt.getScan() / maxTime);
          if (raw == true)
            y = y0 - (int) (h0 * pt.getIntensity() * m_3dGain / m_maxIntensity);
          else
            y = y0
                - (int) (h0 * pt.getSmoothedIntensity() * m_3dGain / m_maxIntensity);
          p.addPoint(x, y);
        }
        p.addPoint(x, y0);
        if ((i % 2) == 0)
          gx.setColor(Color.WHITE);
        else
          gx.setColor(new Color(230, 230, 230));
        if (i == m_curIndex)
          gx.setColor(Color.YELLOW);
        gx.fillPolygon(p);

        // =========================================================================================

//        if (this.m_set.getRawData()==null){
        paint3dAreas(cr, this.m_userProbes, x0, y0, Color.GREEN);
        paint3dAreas(cr, paintableProbes, x0, y0,
            Color.RED);
//        }
        // =========================================================================================

        gx.setColor(Color.BLACK);
        gx.drawPolygon(p);

        // If we are the current diagram, we flag out the daltons:

        if (i == m_curIndex) {
          s = CgUtilities.Format(cr.getMz(), 2) + "d";
          sw = fm.stringWidth(s);
          gx.setColor(Color.YELLOW);
          gx.fillRect(x0 + w0 - sw - 10, y0 - 14, sw + 10, 14);
          gx.setColor(Color.BLACK);
          gx.drawRect(x0 + w0 - sw - 10, y0 - 14, sw + 10, 14);
          gx.drawString(s, x0 + w0 - sw - 5, y0 - 3);
        }

        x0 -= DELTA_X;
        y0 += DELTA_Y;
      }
//      }else{
/*        Vector chromas = m_set.getChromatograms();
        float lowestMz =  ((CgChromatogramVO)chromas.get(0)).getMz();
        float mzValue =  ((CgChromatogramVO)chromas.get(2)).getMz();
        float charge =  this.m_charge;*/
/****        System.out.println(mzValue);
        System.out.println(charge);
        Vector thresholds = new Vector();        
        int count = 0;
        for (Double intensity :m_viewer.idealIntensityDistribution){
          System.out.println("Intensity: "+intensity);
          thresholds.add(mzValue+(1*count)/charge);
          count++;
        }*/
        
/*        float[][] intensities = new float[m_set.getRawData().length/50][m_set.getNumberOfScans()];
        int DELTA_X_NEW = 2;
        int DELTA_Y_NEW = 2;
        float adding = 0.05f;
        w0 = m.getWidth() - leftMargin_3d() - DELTA_X_NEW * (m_set.getRawData().length/50 - 1)
        - rightMargin_3d();
        h0 = m.getHeight() - topMargin_3d() - DELTA_Y_NEW * (m_set.getRawData().length/50 - 1)
          - bottomMargin_3d();

        x0 = leftMargin_3d() + DELTA_X_NEW * (m_set.getRawData().length/50 - 1);
        y0 = m.getHeight() - DELTA_Y_NEW * (m_set.getRawData().length/50 - 1) - bottomMargin_3d();
        xx = x0;
        yy = y0;

        float[] intensities = null;
        len = m_set.getNumberOfScans();
        j=-1;
        int thresholdCount = 0;
        for (i=0; i!=m_set.getRawData().length;i++){
          if (i%50==0){
            j++;
            intensities = new float[m_set.getNumberOfScans()];
          }

          if (m_set.getRawData()[i]!=null&&m_set.getRawData()[i].length()>0){
            ByteBuffer buffer = ByteBuffer.wrap(Base64.decode(m_set.getRawData()[i]));
            while (buffer.hasRemaining()){
              int scanNumber = buffer.getInt();
              float intensity = buffer.getFloat();
////              intensities[j][scanNumber] += intensity;
              intensities[scanNumber] += intensity;
            }
          }
          if (i%50==49){
            
            
            this.drawCurve(gx, len, x0, y0, w0, h0, intensities);

            lowestMz = lowestMz+adding;
            if (thresholds.size()>thresholdCount){
            }
            if (thresholds.size()>thresholdCount&&lowestMz>(Float)thresholds.get(thresholdCount)){
              thresholdCount++;
              gx.setColor(Color.RED);
              maxTime = (Float)m_set.getRetentionTimes().get(len-1);
              x = x0 + (int) (w0 * 0.58*len / maxTime);
              int minusValue = (int) (m_viewer.idealIntensityDistribution.get(thresholdCount-1)* h0 * (m_maxIntensity*0.140) * m_3dGain / m_maxIntensity);
              gx.drawRect(x, y0-minusValue, 1, minusValue );
              
            }
            
            x0 -= DELTA_X_NEW;
            y0 += DELTA_Y_NEW;
          }
          
        }*/
/*        MSMapViewer viewer = MSMapViewerFactory.getMSMapViewer(m_set.getRawData(), m_set.getRetentionTimes());
        viewer.init();
        Rectangle bounds = m_viewer.m_3dPnl.getBounds();
        m_viewer.remove(m_3dPnl);
        viewer.setBounds(bounds);
        m_viewer.add(viewer);*/
        
//        this.m_3dPnl.add(viewer,BorderLayout.CENTER);
        //m.add(viewer);
/****        Map3dViewer viewer = new Map3dViewer(m_viewer,intensities);
        m.add(viewer);
        viewer.setVisible(true);*/

//      }

      // **** Draw the X Coordinate ****

      float t = 0;
      float dt;

      x0 = leftMargin_3d();
      y0 = m.getHeight() - bottomMargin_3d();
      dt = 100;
      t = (m_maxTime - m_minTime) / 10;
      if (t < 100)
        dt = 100;
      else if (t < 200)
        dt = 200;
      else if (t < 500)
        dt = 500;
      else if (t < 1000)
        dt = 1000;
      else if (t < 2000)
        dt = 2000;
      else if (t < 5000)
        dt = 5000;
      else if (t < 10000)
        dt = 10000;
      else if (t < 20000)
        dt = 20000;
      else if (t < 50000)
        dt = 50000;
      else
        dt = 100000;

      j = fm.stringWidth("Time / s");
      t = 0;
      while (t < m_maxTime) {
        if (t >= m_minTime) {
          x = x0 + (int) (w0 * t / (m_maxTime - m_minTime));
          y = y0 + 2;
          gx.drawLine(x, y, x, y + 3);
          s = Float.toString(t);
          sw = fm.stringWidth(s);

          if ((x + sw / 2 + 10) < (x0 + w0 - j))
            gx.drawString(s, x - sw / 2, y + 12);
        }
        t += dt;
      }
      gx.drawString("Time / s", x0 + w0 - j, y + 12);

      m.setBackmostMz(m_minMz);
      m.setFrontmostMz(m_maxMz);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      System.out.println("Draw3DDiagram ===> " + ex.toString());
    }
  }
  
  
  private Vector getPaintableSystemProbes(){
    Vector paintableProbes = new Vector();
    for (int i = 0; m_set==null||i != m_set.getCgProbes().size(); i++) {
      CgProbe aProbe = (CgProbe) m_set.getCgProbes().get(i);
      boolean removed = false;
      for (int j = 0; j != this.removedSystemProbes.size(); j++) {
        if (this.sameCgProbe((CgProbe) this.removedSystemProbes.get(j), aProbe)) {
          removed = true;
        }
      }
      if (!removed)
        paintableProbes.add(aProbe);
    }
    return paintableProbes;
  }
  
  private void drawCurve(Graphics gx, int len, int x0, int y0, int w0, int h0,float[] intensities){
    int x = 0;
//    System.out.println("I am drawing the curve");
    float maxTime = (Float)m_set.getRetentionTimes().get(len-1);
    Polygon p = new Polygon();
    p.addPoint(x0, y0);
    for (int j = 0; j < len; j++) {
//      pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(j);
      x = x0 + (int) (w0 * j / maxTime);
//      if (intensities[j]>0){
//        System.out.println(intensities[j]);
//      }
////      if (raw == true)
        int y = y0 - (int) (h0 * intensities[j] * m_3dGain / m_maxIntensity);
////      else
////        y = y0
////            - (int) (h0 * pt.getSmoothedIntensity() * m_3dGain / m_maxIntensity);
      p.addPoint(x, y);
    }
    p.addPoint(x, y0);
//    if ((i % 2) == 0)
//      gx.setColor(Color.WHITE);
//    else
    gx.setColor(new Color(230, 230, 230));
//    if (i == m_curIndex)
//      gx.setColor(Color.YELLOW);
/*    for (int i=0; i!=p.xpoints.length;i++){
      System.out.println("X: "+p.xpoints[i]+"; Y: "+p.ypoints[i]);
    }*/
    gx.fillPolygon(p);

    // =========================================================================================

    // =========================================================================================

    gx.setColor(Color.BLACK);
    gx.drawPolygon(p);
   
  }

  private void paint3dAreas(CgChromatogramVO cr, Vector areas, int x0, int y0,
      Color col)
  {
    int i, j;
    Polygon p;
    CgChromatogramPointVO pt;
    int x = 0, y = 0;

    int w0 = diagramWidth_3d();
    int h0 = diagramHeight_3d();

    Graphics gx = m_3dPnl.getBufferGraphics();
    int len = cr.getChromatogramPointVOs().size();
    pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(len - 1);
    float maxTime = pt.getScan();
    for (i = 0; i < areas.size(); i++) {
      CgProbe cp = (CgProbe) areas.get(i);
      if (cp == null)
        continue;
      if (cp.AreaStatus != CgAreaStatus.OK)
        continue;

      if (cp.Mz < (cr.getMz() + this.getThreshold())
          && cp.Mz > (cr.getMz() - this.getThreshold())) {
        m_3dPnl.addArea(cp.Area);

        // Paint the Area of the Peak accordingly:

        p = new Polygon();
        boolean firstPoint2 = true;
        for (j = 0; j < len; j++) {
          pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(j);

          if (pt.getScan() < m_minDispTime3d)
            continue;
          if (pt.getScan() < cp.LowerValley)
            continue;
          if (pt.getScan() > m_maxDispTime3d)
            continue;
          if (pt.getScan() > cp.UpperValley)
            continue;

          x = x0 + (int) (w0 * pt.getScan() / maxTime);
          if (m_raw == true)
            y = y0 - (int) (h0 * pt.getIntensity() * m_3dGain / m_maxIntensity);
          else
            y = y0
                - (int) (h0 * pt.getSmoothedIntensity() * m_3dGain / m_maxIntensity);

          if (firstPoint2 == true) {
            p.addPoint(x, y0);
            firstPoint2 = false;
          }
          p.addPoint(x, y);
        }
        if (p.npoints > 0) {
          p.addPoint(x, y0);
          gx.setColor(col);
          gx.fillPolygon(p);
        }
      }
    }

  }

  private boolean sameCgProbe(CgProbe probe1, CgProbe probe2)
  {
    boolean ok = false;
    /**** Here it is questionable if the CgProbe.Peak should be compared (for enhanced valley method) because the peak
     * can shift and then the area is painted green instead of red
     */
    if (probe1.Area >= (probe2.Area - 0.02)
        && probe1.Area <= (probe2.Area + 0.02)
        && probe1.AreaError >= (probe2.AreaError - 0.02)
        && probe1.AreaError <= (probe2.AreaError + 0.02)
        && probe1.Background >= (probe2.Background - 0.02)
        && probe1.Background <= (probe2.Background + 0.02)
        && probe1.Charge == probe2.Charge
        && probe1.LowerMzBand >= (probe2.LowerMzBand - this.getThreshold())
        && probe1.LowerMzBand <= (probe2.LowerMzBand + this.getThreshold())
        && probe1.LowerValley >= (probe2.LowerValley - 0.02)
        && probe1.LowerValley <= (probe2.LowerValley + 0.02)
        && probe1.Mz >= (probe2.Mz - this.getThreshold())
        && probe1.Mz <= (probe2.Mz + this.getThreshold())
        && probe1.Peak >= (probe2.Peak - 0.02)
        && probe1.Peak <= (probe2.Peak + 0.02)
        && probe1.UpperMzBand >= (probe2.UpperMzBand - this.getThreshold())
        && probe1.UpperMzBand <= (probe2.UpperMzBand + this.getThreshold())
        && probe1.UpperValley >= (probe2.UpperValley - 0.02)
        && probe1.UpperValley <= (probe2.UpperValley + 0.02)) {
      ok = true;
    }
    return ok;
  }

  private void createImage()
  {
    int x, y, len;
    int r, g, b;
    CgChromatogramVO cr;
    CgChromatogramPointVO pt;

    if (m_maxPoints == 0 || m_chromaCount == 0)
      return;
    m_img = m_viewer.createImage(m_maxPoints, m_chromaCount);
    Graphics gx = m_img.getGraphics();
    g = 0;
    for (y = 0; y < m_chromaCount; y++) {
      cr = (CgChromatogramVO) m_set.getChromatograms().get(y);
      len = cr.getChromatogramPointVOs().size();
      for (x = 0; x < len; x++) {
        pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(x);
        r = (int) (255 * (pt.getSmoothedIntensity() / m_maxIntensity));
        b = 255 - r;
        gx.setColor(new Color(b, b, b));
        gx.drawLine(x, y, x, y);
      }
    }
    int yy = y * 8;
    if (yy > 300)
      yy = 300;

    m_img = m_img.getScaledInstance(1000, yy, Image.SCALE_SMOOTH);
  }

  private void calcBoundaries()
  {
    float val;
    CgChromatogramVO cr;
    CgChromatogramPointVO pt;
    int len;

    if (m_set == null)
      return;
    if (m_set.getChromatograms() == null)
      return;

    m_maxPoints = 0;
    m_minMz = 999999999;
    m_minRawMz = 999999999;
    m_maxMz = 0;
    m_maxRawMz = 0;
    m_minTime = 999999999;
    m_maxTime = 0;
    m_maxIntensity = 0;
    m_maxIntensitySmooth = 0;
    m_maxIntensityRaw = 0;

    m_chromaCount = m_set.getChromatograms().size();
    for (int i = 0; i < m_chromaCount; i++) {
      cr = (CgChromatogramVO) m_set.getChromatograms().get(i);
      val = cr.getMz();
      if (val < m_minMz){
        m_minMz = val;
      }  
      if (val > m_maxMz)
        m_maxMz = val;

      len = cr.getChromatogramPointVOs().size();
      if (m_maxPoints < len)
        m_maxPoints = len;
      for (int j = 0; j < len; j++) {
        pt = (CgChromatogramPointVO) cr.getChromatogramPointVOs().get(j);
        val = pt.getScan();
        if (val < m_minTime)
          m_minTime = val;
        if (val > m_maxTime)
          m_maxTime = val;
        val = pt.getIntensity();
        if (val > m_maxIntensity)
          m_maxIntensity = val;
        if (val > m_maxIntensityRaw)
          m_maxIntensityRaw = val;
        val = pt.getSmoothedIntensity();
        if (val > m_maxIntensitySmooth)
          m_maxIntensitySmooth = val;
      }
    }
    float stepSize = ((m_maxMz-m_minMz)/(m_chromaCount-1));
    this.m_minRawMz = m_minMz-0.5f*stepSize;
    this.m_maxRawMz = m_maxMz+0.5f*stepSize;
    m_minDispTime3d = m_minTime;
    m_maxDispTime3d = m_maxTime;
  }

  // =============================================================================================
  // Helpers for the 2d Panel:
  // =============================================================================================

  private int leftMargin_2d()
  {
    return 75;
  }

  private int rightMargin_2d()
  {
    return 15;
  }

  private int diagramWidth_2d()
  {
    return m_2dPnl.getWidth() - leftMargin_2d() - rightMargin_2d();
  }

  private int topMargin_2d()
  {
    return 5;
  }

  private int bottomMargin_2d()
  {
    return 20;
  }

  private int diagramHeight_2d()
  {
    return m_2dPnl.getHeight() - topMargin_2d() - bottomMargin_2d();
  }

  private boolean in2dDiagram(int x, int y)
  {
    if (x < leftMargin_2d())
      return false;
    if (x > (leftMargin_2d() + diagramWidth_2d()))
      return false;
    if (y < topMargin_2d())
      return false;
    if (y > (topMargin_2d() + diagramHeight_2d()))
      return false;
    return true;
  }

  // =============================================================================================
  // Helpers for the 3d Panel:
  // =============================================================================================

  private int diagramDeltaX_3d()
  {
    return 4;
  }

  private int diagramDeltaY_3d()
  {
    return 6;
  }

  private int leftMargin_3d()
  {
    return 5;
  }

  private int rightMargin_3d()
  {
    return 15;
  }

  private int topMargin_3d()
  {
    return 10;
  }

  private int bottomMargin_3d()
  {
    return 20;
  }

  private int diagramWidth_3d()
  {
    return m_3dPnl.getWidth() - leftMargin_3d() - diagramDeltaX_3d()
        * (m_chromaCount - 1) - rightMargin_3d();
  }

  private int diagramHeight_3d()
  {
    return m_3dPnl.getHeight() - topMargin_3d() - diagramDeltaY_3d()
        * (m_chromaCount - 1) - bottomMargin_3d();
  }

  private double getThreshold()
  {
    return ((new Double(this.m_viewer.m_stepMzText.getText())).doubleValue() / 50);
  }

  public CgChromatogramSet getM_set()
  {
    return m_set;
  }

  public float getM_maxMz()
  {
    return m_maxMz;
  }

  public float getM_minMz()
  {
    return m_minMz;
  }

  public float getM_maxRawMz()
  {
    return m_maxRawMz;
  }

  public float getM_minRawMz()
  {
    return m_minRawMz;
  }
  
  public float[] getCurrentMzRange(){
    float[] rangeValues = new float[2];
    if (m_set!=null){
      CgChromatogramVO cx = (CgChromatogramVO) m_set.getChromatograms().get(this.m_curIndex);
      rangeValues[0] = cx.getMz()-cx.getLowerMzBand();
      rangeValues[1] = cx.getMz()+cx.getUpperMzBand();
    }
    return rangeValues;
  }
  
  public Vector getProbesToPaint(){
    Vector allProbes = new Vector();
    allProbes.add(this.getPaintableSystemProbes());
    allProbes.add(this.m_userProbes);
    return allProbes;
  }
  
  public void destroyInfoObject(){
    this.m_2dPnl.destroy2dGraphicsManager();
    this.m_2dPnl.getGraphics().dispose();
    this.m_2dPnl = null;
    this.m_3dPnl.destroy3dGraphicsManager();
    if (this.m_3dPnl.getGraphics()!=null)
      this.m_3dPnl.getGraphics().dispose();
    this.m_3dPnl = null;
    if (this.m_img!=null){
      this.m_img.getGraphics().dispose();
      this.m_img.flush();
    }
    this.m_img = null;
    this.m_set = null;
    this.m_userProbes = null;
    if (this.m_viewer.getGraphics()!=null)
      this.m_viewer.getGraphics().dispose();
    this.m_viewer = null;
    this.removedSystemProbes = null;
//    try {
//      this.finalize();
//    }
//    catch (Throwable e) {
      // TODO Auto-generated catch block
//      e.printStackTrace();
//    }
  }
}
